/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.trash;

import java.util.Set;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.plugins.repository.jcr.JCRTraversableAmetysObject;

/**
 * An {@link AmetysObject} that represents an other {@link AmetysObject} that was trashed
 */
public interface TrashElement extends JCRTraversableAmetysObject, ModifiableModelAwareDataAwareAmetysObject
{
    /**
     * Get the id of the trashed Ametys object
     * @return the id
     */
    String getAmetysObjectId();

    /**
     * Get the trash element that represent one of the linked objects (see {@link #addLinkedObjects}) of this element
     * @param hiddenOnly true to only retrieve the hidden elements
     * @return the linked elements
     */
    Set<TrashElement> getLinkedElements(boolean hiddenOnly);
    
    /**
     * Add the objects to the list of objects whose thrashing was related to the trashing
     * of this element's object. If an element representing these linked objects in the trash
     * is present during restoration or deletion of the current element, they should be restored/deleted
     * at the same time.
     * @param linkedObjectsIds a list of Ametys object ids
     */
    void addLinkedObjects(String... linkedObjectsIds);

    /**
     * Set the hidden status of the trash element.<br>
     * 
     * An element should be hidden if it was trashed because an other object
     * was trashed.<br>
     * Hidden element should not be visible when listing the trash elements,
     * and should be restored or permanently deleted along with the element that
     * triggered the thrashing
     * @param hidden true to set hidden
     */
    void setHidden(boolean hidden);
}
