/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.trash;


import java.util.List;
import java.util.Map;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ItemParserHelper.ConfigurationAndPluginName;
import org.ametys.runtime.plugin.component.Prioritizable;
import org.ametys.runtime.plugin.component.Supporter;
/**
 * An extension supporting some {@link TrashableAmetysObject} and providing informations related to the object
 */
public interface TrashElementType extends Prioritizable, Supporter<TrashableAmetysObject>
{

    /**
     * Get the id of the extension in the {@link TrashElementTypeExtensionPoint}
     * @return the id
     */
    String getId();
    
    /**
     * Get the label describing the type of object supported by this type
     * @return the label
     */
    I18nizableText getLabel();
    
    /**
     * Get a icon glyph for a trash element
     * @param element the trash element
     * @return the css class for the glyph
     */
    Map<String, String> getIcon(TrashElement element);
    
    /**
     * Get the message target corresponding to the object
     * @param object the object
     * @return the message target type of the object
     */
    String getMessageTargetType(TrashableAmetysObject object);
    
    /**
     * Get a description describing the result of the restoration of this object
     * @param object the restored object
     * @return the description
     */
    I18nizableText getRestorationDescription(TrashableAmetysObject object);
    
    /**
     * Get the necessary information to add a open tool action in the restoration notification
     * @param restoredObject the restored object
     * @return a map with the tool id and and parameters
     */
    Map<String, Object> getNotificationOpenToolAction(TrashableAmetysObject restoredObject);
    
    /**
     * Additional action to perform after restoring an supported object
     * @implSpec This method should be called after all linked elements are restored to be able
     * to perform relation checks and other such operation.
     * @param restoredObject the restored object
     */
    void additionnalRestoreAction(TrashableAmetysObject restoredObject);

    /**
     * Get a list of model items to be included in the trash element model
     * @return the list of model items to include
     */
    List<ConfigurationAndPluginName> getAdditionalModelItemsConfigurationAndPluginName();
}
