/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.version;

import java.util.Date;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * {@link AmetysObject} that is versioned and knows about its revisions and labels.
 */
public interface VersionAwareAmetysObject extends AmetysObject
{
    /**
     * List all revisions of this object. Revisions are ordered in chronological order.
     * @return the list of revisions (can be empty, but not <code>null</code>)
     * @throws AmetysRepositoryException if the revisions cannot be known.
     */
    public String[] getAllRevisions() throws AmetysRepositoryException;

    /**
     * List all labels that exist on this object, whatever the revision.
     * @return the list of all labels (can be empty, but not <code>null</code>).
     * @throws AmetysRepositoryException if the labels cannot be known.
     */
    public String[] getAllLabels() throws AmetysRepositoryException;

    /**
     * Get the revision of this object, if any.
     * @return the revision or null if this is the latest.
     * @throws AmetysRepositoryException if the revision cannot be known.
     */
    public String getRevision() throws AmetysRepositoryException;

    /**
     * Get the creation time of the current revision, if any.
     * @return the revision creation time, or null if there's no current revision.
     * @throws AmetysRepositoryException if an error occurs
     */
    public Date getRevisionTimestamp() throws AmetysRepositoryException;

    /**
     * Get the creation time of the given revision.
     * @param revision the revision.
     * @return the revision creation date.
     * @throws UnknownAmetysObjectException if the given revision does not exist for this object.
     * @throws AmetysRepositoryException if an error occurs
     */
    public Date getRevisionTimestamp(String revision) throws AmetysRepositoryException;

    /**
     * Get the labels for this object, in the current revision (a single revision can hold several labels).
     * @return the list of labels for this revision (can be empty, but not <code>null</code>).
     * @throws AmetysRepositoryException if the labels cannot be known.
     */
    public String[] getLabels() throws AmetysRepositoryException;

    /**
     * Get the labels for this object fot the given revision (a single revision can hold several labels).
     * @param revision the revision
     * @return the list of labels for this revision (can be empty, but not <code>null</code>).
     * @throws UnknownAmetysObjectException if the given revision does not exist for this object.
     * @throws AmetysRepositoryException if the labels cannot be known.
     */
    public String[] getLabels(String revision) throws UnknownAmetysObjectException, AmetysRepositoryException;
    
    /**
     * Switch to the revision corresponding to the specified label.<br>
     * All subsequent method calls on this {@link AmetysObject} will concern that version.
     * @param label the label to switch to, or null to specify the current version
     * @throws UnknownAmetysObjectException if the label does not correspond to any revision
     * @throws AmetysRepositoryException if a problem occurs
     */
    public void switchToLabel(String label) throws UnknownAmetysObjectException, AmetysRepositoryException;

    /**
     * Switch to the revision corresponding to the specified revision.<br>
     * All subsequent method calls on this {@link AmetysObject} will concern that version.
     * @param revision the revision, or null to specify the current version
     * @throws UnknownAmetysObjectException if the revision does not exist
     * @throws AmetysRepositoryException if a problem occurs
     */
    public void switchToRevision(String revision) throws UnknownAmetysObjectException, AmetysRepositoryException;
}
