/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.workspace;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.lock.LockAwareAmetysObject;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.repository.metadata.MetadataAwareAmetysObject;
import org.ametys.plugins.repository.metadata.ModifiableCompositeMetadata;
import org.ametys.plugins.repository.metadata.ModifiableMetadataAwareAmetysObject;

/**
 * DAO providing methods to manage {@link AmetysObject}s.
 */
public class AmetysObjectDao extends AbstractLogEnabled implements Component, Serviceable
{
    
    /** The AmetysObject resolver. */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get information on an {@link AmetysObject} from its identifier.
     * @param id the AmetysObject identifier.
     * @return information on the AmetysObject as a Map.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> getAmetysObject(String id)
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Getting AmetysObject from ID '" + id + "'");
        }
        
        AmetysObject obj = _resolver.resolveById(id);
        Map<String, Object> info = new HashMap<>();
        putAmetysObjectInfo(info, obj);
        
        return info;
    }
    
    /**
     * Get information on an {@link AmetysObject}, found by its path.
     * @param path the absolute AmetysObject path (starting with /ametys:root).
     * @return information on the AmetysObject as a Map.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> getAmetysObjectByPath(String path)
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Getting AmetysObject at path '" + path + "'");
        }
        
        AmetysObject obj = _resolver.resolveByPath(path);
        Map<String, Object> info = new HashMap<>();
        putAmetysObjectInfo(info, obj);
        
        return info;
    }
    
    /**
     * Get information on a {@link AmetysObject}s from their identifier.
     * @param ids a Collection of AmetysObject identifiers.
     * @return a Map containing the list of AmetysObject info ('objects' key)
     * and the list of unknown identifiers ('notFound' key).
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> getAmetysObjects(Collection<String> ids)
    {
        List<Map<String, Object>> objects = new ArrayList<>();
        List<String> notFound = new ArrayList<>();
        
        for (String id : ids)
        {
            try
            {
                AmetysObject obj = null;
                if ("/".equals(id))
                {
                    obj = _resolver.resolveByPath("/");
                }
                else
                {
                    obj = _resolver.resolveById(id);
                }
                
                Map<String, Object> objectInfo = new HashMap<>();
                putAmetysObjectInfo(objectInfo, obj);
                objects.add(objectInfo);
            }
            catch (UnknownAmetysObjectException e)
            {
                notFound.add(id);
            }
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("objects", objects);
        result.put("notFound", notFound);
        
        return result;
    }
    
    /**
     * Add an AmetysObject to the repository.
     * @param parentId the parent AmetysObject ID, must be ModifiableTraversable.
     * @param name the name of the AmetysObject to create.
     * @param type the AmetysObject type.
     * @return a Map of information on the created AmetysObject.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> addAmetysObject(String parentId, String name, String type)
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Trying to add child: '" + name + "' to the AmetysObject of id '" + parentId + "'");
        }
        
        AmetysObject obj = null;
        if (parentId.equals("/"))
        {
            obj = _resolver.resolveByPath("/");
        }
        else
        {
            obj = _resolver.resolveById(parentId);
        }
        
        if (!(obj instanceof ModifiableTraversableAmetysObject))
        {
            throw new IllegalArgumentException();
        }
        
        // Create child if modifiable traversable.
        AmetysObject child = ((ModifiableTraversableAmetysObject) obj).createChild(name, type);
        ((ModifiableTraversableAmetysObject) obj).saveChanges();
        
        Map<String, Object> childInfo = new HashMap<>();
        putAmetysObjectInfo(childInfo, child);
        
        return childInfo;
    }
    
    /**
     * Remove an AmetysObject.
     * @param id the identifier of the AmetysObject to remove.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public void removeAmetysObject(String id)
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Trying to remove the AmetysObject of id '" + id + "'");
        }
        
        AmetysObject obj = null;
        
        if (id.equals("/"))
        {
            obj = _resolver.resolveByPath("/");
        }
        else
        {
            obj = _resolver.resolveById(id);
        }
        
        if (obj instanceof RemovableAmetysObject)
        {
            ModifiableAmetysObject parent = obj.getParent();
            ((RemovableAmetysObject) obj).remove();
            parent.saveChanges();
        }
    }
    
    /**
     * Remove a metadata of a Ametys object
     * @param id the identifier of the AmetysObject
     * @param compositePath the path of the parent metadata. Can be null or empty.
     * @param metadataName The name of metadata to remove
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public void removeMetadata (String id, String compositePath, String metadataName)
    {
        AmetysObject obj = null;
        if (id.equals("/"))
        {
            obj = _resolver.resolveByPath("/");
        }
        else
        {
            obj = _resolver.resolveById(id);
        }
        
        if (obj instanceof ModifiableMetadataAwareAmetysObject)
        {
            ModifiableCompositeMetadata holder = ((ModifiableMetadataAwareAmetysObject) obj).getMetadataHolder();
            
            if (StringUtils.isNotEmpty(compositePath))
            {
                // deep search in composite metadata
                if (!compositePath.isEmpty())
                {
                    String[] tokens = compositePath.split("/");        
                    for (int i = 0; i < tokens.length; i++)
                    {
                        holder = holder.getCompositeMetadata(tokens[i]);
                    }
                }
            }
            
            holder.removeMetadata(metadataName);
            ((ModifiableMetadataAwareAmetysObject) obj).saveChanges();
        }
    }
    
    /**
     * Unlock an AmetysObject.
     * @param id the identifier of the AmetysObject to remove.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public void unlockAmetysObject(String id)
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Trying to unlock the AmetysObject of id '" + id + "'");
        }
        
        AmetysObject obj = null;
        
        if (id.equals("/"))
        {
            obj = _resolver.resolveByPath("/");
        }
        else
        {
            obj = _resolver.resolveById(id);
        }
        
        if (obj instanceof LockableAmetysObject)
        {
            LockableAmetysObject lockableObject = (LockableAmetysObject) obj;
            if (lockableObject.isLocked())
            {
                lockableObject.unlock();
            }
        }
    }
    
    /**
     * Execute a query to find AmetysObjects.
     * @param query the query to execute.
     * @return a List of AmetysObject info (as Maps).
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public List<Map<String, Object>> query(String query)
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Executing logic query: " + query);
        }
        
        List<Map<String, Object>> results = new ArrayList<>();
        
        if (StringUtils.isNotBlank(query))
        {
            AmetysObjectIterable<AmetysObject> objects = _resolver.query(query);
            for (AmetysObject object : objects)
            {
                Map<String, Object> result = new HashMap<>();
                result.put("id", object.getId());
                result.put("name", object.getName());
                result.put("path", object.getPath());
                
                results.add(result);
            }
        }
        
        return results;
    }
    
    /**
     * Fill the given Map with information on an AmetysObject.
     * @param info the Map to fill.
     * @param obj the AmetysObject.
     */
    protected void putAmetysObjectInfo(Map<String, Object> info, AmetysObject obj)
    {
        try
        {
            info.put("id", obj.getId());
            info.put("name", obj.getName());
            info.put("path", obj.getPath());
            
            info.put("metadataAware", obj instanceof MetadataAwareAmetysObject);
            info.put("modifiable", obj instanceof ModifiableAmetysObject);
            info.put("modifiableTraversable", obj instanceof ModifiableTraversableAmetysObject);
            
            if (obj instanceof JCRAmetysObject)
            {
                info.put("jcrAo", true);
                info.put("jcrPath", ((JCRAmetysObject) obj).getNode().getPath());
            }
            
            if (obj instanceof LockAwareAmetysObject)
            {
                info.put("locked", ((LockAwareAmetysObject) obj).isLocked());
                info.put("lockable", obj instanceof LockableAmetysObject);
            }
            else
            {
                info.put("lockable", false);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting information on the object " + obj.toString(), e);
        }
    }
    
}
