/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.workspace;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.reading.AbstractReader;
import org.apache.cocoon.reading.Reader;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceUtil;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.metadata.BinaryMetadata;
import org.ametys.plugins.repository.metadata.CompositeMetadata;
import org.ametys.plugins.repository.metadata.File;
import org.ametys.plugins.repository.metadata.Folder;
import org.ametys.plugins.repository.metadata.ModifiableMetadataAwareAmetysObject;
import org.ametys.plugins.repository.metadata.RichText;

/**
 * {@link Reader} for exposing a binary property from a JCR repository.<br>
 * Mime type can be forced by using parameter <code>mime-type</code>.
 */
public class BinaryMetadataReader extends AbstractReader implements Serviceable
{
    private AmetysObjectResolver _resolver;
    private String _mimeType;
    private String _mdName;
    private String _compositePath;
    private String _richtextPath;
    private String _mdType;
    private String _aoId;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void setup(SourceResolver res, Map objModel, String src, Parameters params) throws ProcessingException, SAXException, IOException
    {
        super.setup(res, objModel, src, params);
        
        try
        {
            _mimeType = params.getParameter("mime-type");
            _mdName = params.getParameter("name");
            _compositePath = params.getParameter("compositePath");
            _richtextPath = params.getParameter("richtextPath");
            _mdType = params.getParameter("type");
            _aoId = params.getParameter("id");
        }
        catch (ParameterException e)
        {
            throw new ProcessingException(e);
        }
    }
    
    @Override
    public String getMimeType()
    {
        if (!StringUtils.isEmpty(_mimeType))
        {
            return _mimeType;
        }
        
        return _getMimeType();
    }

    @SuppressWarnings({"resource", "deprecation"})
    public void generate() throws IOException, SAXException, ProcessingException
    {
        try
        {
            String filename = null;
            long length = 0;
            
            AmetysObject ao = _getAmetysObject();
            
            if (getLogger().isInfoEnabled())
            {
                getLogger().info("Trying to read a binary for object of id: '" + _aoId + "'");
            }
            
            CompositeMetadata holder = _getCompositeMetadata (ao);
            InputStream is = null;
            
            if (_mdType.equals("RICHTEXT-CONTENT"))
            {            
                String[] rtTokens = _richtextPath.split("/");
                RichText rt = holder.getRichText(rtTokens[0]);
                
                length = rt.getLength();
                filename = rtTokens[0];
                is = rt.getInputStream();
            }
            else if (_mdType.equals("RICHTEXT"))
            {
                RichText rt = holder.getRichText(_mdName);
                
                length = rt.getLength();
                filename = _mdName;
                is = rt.getInputStream();
            }
            else if (_mdType.equals("BINARY"))
            {
                BinaryMetadata binaryMetadata = holder.getBinaryMetadata(_mdName);
                filename = binaryMetadata.getFilename();
                is = binaryMetadata.getInputStream();
                length = binaryMetadata.getLength();
            }
            else if (_mdType.equals("FILE-CONTENT"))
            {
                //TODO this case is not tested
                String[] rtTokens = _richtextPath.split("/");
                RichText rt = holder.getRichText(rtTokens[0]);
                length = rt.getLength();
                
                Folder folder = rt.getAdditionalDataFolder();            
                for (int i = 2; i < rtTokens.length - 1; i++)
                {
                    folder = folder.getFolder(rtTokens[i]);
                }
                
                File file = folder.getFile(rtTokens[rtTokens.length - 1]);
                filename = file.getName();
                is = file.getResource().getInputStream();
            }


            Response response = ObjectModelHelper.getResponse(objectModel);
            
            response.setHeader("Content-Length", Long.toString(length));
            response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
            
            SourceUtil.copy(is, out);
        }
        catch (Exception e)
        {
            throw new ProcessingException("Unable to download binary property of path " + _mdName, e);
        }
        finally
        {
            IOUtils.closeQuietly(out);
        }
    }
    
    private CompositeMetadata _getCompositeMetadata (AmetysObject ao)
    {
        CompositeMetadata holder = ((ModifiableMetadataAwareAmetysObject) ao).getMetadataHolder();
        
        // deep search if inside composite metadata
        if (!_compositePath.isEmpty())
        {
            String[] tokens = _compositePath.split("/"); 
            for (String token : tokens)
            {
                holder = holder.getCompositeMetadata(token);
            }
        }  
        
        return holder;
    }
    
    private String _getMimeType ()
    {
        AmetysObject ao = _getAmetysObject();
        CompositeMetadata holder = _getCompositeMetadata (ao);
        
        if (_mdType.equals("RICHTEXT-CONTENT"))
        {            
            String[] rtTokens = _richtextPath.split("/");
            RichText rt = holder.getRichText(rtTokens[0]);
            return rt.getMimeType();
        }
        else if (_mdType.equals("RICHTEXT"))
        {
            return holder.getRichText(_mdName).getMimeType();
        }
        else if (_mdType.equals("BINARY"))
        {
            return holder.getBinaryMetadata(_mdName).getMimeType();
        }
        else if (_mdType.equals("FILE-CONTENT"))
        {
            String[] rtTokens = _richtextPath.split("/");
            RichText rt = holder.getRichText(rtTokens[0]);
            
            Folder folder = rt.getAdditionalDataFolder();            
            for (int i = 2; i < rtTokens.length - 1; i++)
            {
                folder = folder.getFolder(rtTokens[i]);
            }
            
            File file = folder.getFile(rtTokens[rtTokens.length - 1]);
            return file.getResource().getMimeType();
        }
        
        return "application/unknown";
    }
    
    private AmetysObject _getAmetysObject ()
    {
        if (_aoId.equals("/"))
        {
            return _resolver.resolveByPath("/");
        }
        else
        {
            return _resolver.resolveById(_aoId);
        }
    }
}
