/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.workspace;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.lock.LockAwareAmetysObject;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.repository.metadata.CompositeMetadata;
import org.ametys.plugins.repository.metadata.CompositeMetadata.MetadataType;
import org.ametys.plugins.repository.metadata.File;
import org.ametys.plugins.repository.metadata.Folder;
import org.ametys.plugins.repository.metadata.MetadataAwareAmetysObject;
import org.ametys.plugins.repository.metadata.RichText;
import org.ametys.workspaces.repository.jcr.NodeGroupHelper;

/**
 * Generate the content of a logic node:
 * <ul>
 *  <li>metadata
 *  <li>ametys object properties
 *  <li>children
 * </ul>
 */
public class RepositoryLogicNodeGenerator extends ServiceableGenerator
{
    private AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Trying to generate content for a node");
        }

        String id = parameters.getParameter("id", "");
        
        contentHandler.startDocument();
        AttributesImpl repositoryAttributes = new AttributesImpl();

        XMLUtils.startElement(contentHandler, "repository", repositoryAttributes);

        try
        {
            _saxObject(id);
        }
        catch (RepositoryException e)
        {
            throw new ProcessingException("Unable to retrieve node for path: '" + source + "'", e);
        }

        XMLUtils.endElement(contentHandler, "repository");

        contentHandler.endDocument();
    }
    
    private void _saxObject(String id) throws SAXException, RepositoryException
    {
        AmetysObject ao = null;
        
        if (StringUtils.isEmpty(id) || id.equals("/"))
        {
            ao = _resolver.resolveByPath("/");
        }
        else
        {
            ao = _resolver.resolveById(id);
        }
        
        boolean isModifiable = ao instanceof ModifiableAmetysObject;
        boolean isLocked = ao instanceof LockAwareAmetysObject && ((LockAwareAmetysObject) ao).isLocked();
        AttributesImpl nodeAttributes = new AttributesImpl();

        nodeAttributes.addCDATAAttribute("path", ao.getPath());
        nodeAttributes.addCDATAAttribute("name", ao.getName());
        nodeAttributes.addCDATAAttribute("id", ao.getId());

        XMLUtils.startElement(contentHandler, "ametysObject", nodeAttributes);

        Attributes attrs = XMLUtils.EMPTY_ATTRIBUTES;
        // MetaData
        if (ao instanceof MetadataAwareAmetysObject)
        {
            XMLUtils.createElement(contentHandler, "isMetadataAware", attrs, "true");
            CompositeMetadata holder = ((MetadataAwareAmetysObject) ao).getMetadataHolder();
            _saxMetaData(holder, isLocked, isModifiable);
        }
        else
        {
            XMLUtils.createElement(contentHandler, "isMetadataAware", attrs, "false");
        }

        // Children
        if (ao instanceof TraversableAmetysObject)
        {
            _saxChildren((TraversableAmetysObject) ao);
        }

        if (ao instanceof JCRAmetysObject)
        {
            Node node = ((JCRAmetysObject) ao).getNode();
            XMLUtils.createElement(contentHandler, "isJCRAO", attrs, "true");
            XMLUtils.createElement(contentHandler, "JCRPath", attrs, node.getPath());
            XMLUtils.createElement(contentHandler, "JCRPathWithGroups", attrs, NodeGroupHelper.getPathWithGroups(node));
        }
        else
        {
            XMLUtils.createElement(contentHandler, "isJCRAO", attrs, "false");
        }

        XMLUtils.createElement(contentHandler, "isModifiable", attrs, Boolean.toString(isModifiable));
        XMLUtils.createElement(contentHandler, "isModifiableTraversable", attrs, Boolean.toString(ao instanceof ModifiableTraversableAmetysObject));
        
        if (ao instanceof LockAwareAmetysObject)
        {
            XMLUtils.createElement(contentHandler, "isLocked", attrs, Boolean.toString(isLocked));
            XMLUtils.createElement(contentHandler, "isLockable", attrs, Boolean.toString(ao instanceof LockableAmetysObject));
        }
        else
        {
            XMLUtils.createElement(contentHandler, "isLocked", attrs, "not-applicable");
            XMLUtils.createElement(contentHandler, "isLockable", attrs, "false");
        }
        

        XMLUtils.endElement(contentHandler, "ametysObject");
    }

    private void _saxMetaData(CompositeMetadata holder, boolean isLocked, boolean isModifiable) throws SAXException, RepositoryException
    {        
        String[] metadataNames = holder.getMetadataNames();
        
        List<String> metaNames = new ArrayList<>(Arrays.asList(metadataNames));
        Collections.sort(metaNames);
        
        for (String name : metaNames)
        {
            AttributesImpl metadataAttributes = new AttributesImpl();

            MetadataType type = holder.getType(name);
            Boolean isMultiple = holder.isMultiple(name);

            metadataAttributes.addCDATAAttribute("name", name);
            metadataAttributes.addCDATAAttribute("type", type.toString());
            metadataAttributes.addCDATAAttribute("multiple", Boolean.toString(isMultiple));
            metadataAttributes.addCDATAAttribute("editable", Boolean.toString(type != MetadataType.COMPOSITE && isModifiable && !isLocked));
            
            XMLUtils.startElement(contentHandler, "metadata", metadataAttributes);

            _saxValues(holder, name, type, isMultiple, isLocked, isModifiable);

            XMLUtils.endElement(contentHandler, "metadata");
        }

    }

    private void _saxChildren(TraversableAmetysObject ao) throws SAXException, RepositoryException
    {        
        AmetysObjectIterable<AmetysObject> children = ao.getChildren();
        Iterator<AmetysObject> it = children.iterator();

        while (it.hasNext())
        {
            AmetysObject child = it.next();

            String id = child.getId();
            String name = child.getName();

            AttributesImpl childAttributes = new AttributesImpl();
            childAttributes.addCDATAAttribute("name", name);
            childAttributes.addCDATAAttribute("id", id);
            childAttributes.addCDATAAttribute("path", child.getPath());
            
            if (child instanceof JCRAmetysObject)
            {
                Node node = ((JCRAmetysObject) child).getNode();
                childAttributes.addCDATAAttribute("isJCRAO", "true");
                childAttributes.addCDATAAttribute("JCRPath", node.getPath());
                childAttributes.addCDATAAttribute("JCRPathWithGroups", NodeGroupHelper.getPathWithGroups(node));
            }
            
            // If not traversable or no child
            if (!(child instanceof TraversableAmetysObject))
            {
                childAttributes.addCDATAAttribute("noChild", "true");
            }
            else if (!((TraversableAmetysObject) child).getChildren().iterator().hasNext())
            {
                childAttributes.addCDATAAttribute("noChild", "true");
            }

            XMLUtils.createElement(contentHandler, "ametysObject", childAttributes);
        }
    }

    private void _saxValues(CompositeMetadata holder, String name, MetadataType type, Boolean isMultiple, boolean isLocked, boolean isModifiable) throws RepositoryException, SAXException
    {
        if (isMultiple)
        {
            _saxMultipleValues(holder, name, type, isLocked, isModifiable);
        }
        else
        {
            _saxSimpleValue(holder, name, type, isLocked, isModifiable);
        }
    }

    private void _saxMultipleValues(CompositeMetadata holder, String name, MetadataType type, boolean isLocked, boolean isModifiable) throws RepositoryException, SAXException
    {
        Attributes attrs = XMLUtils.EMPTY_ATTRIBUTES;
        String textValue = null;

        switch (type)
        {
            case DATE:
                Date[] dateValues = holder.getDateArray(name);
                for (Date dVal : dateValues)
                {
                    XMLUtils.createElement(contentHandler, "value", attrs, DateUtils.dateToString(dVal));
                }
                break;
            case STRING:
                String[] stringValues = holder.getStringArray(name);
                for (String sVal : stringValues)
                {
                    textValue = sVal;
                    XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                }
                break;
            case LONG:
                long[] longValues = holder.getLongArray(name);
                for (long lVal : longValues)
                {
                    textValue = Long.toString(lVal);
                    XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                }
                break;
            case DOUBLE:
                double[] doubleValues = holder.getDoubleArray(name);
                for (double dbVal : doubleValues)
                {
                    textValue = Double.toString(dbVal);
                    XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                }
                break;
            case BOOLEAN:
                boolean[] booleanValues = holder.getBooleanArray(name);
                for (boolean bVal : booleanValues)
                {
                    textValue = Boolean.toString(bVal);
                    XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                }
                break;
            case COMPOSITE:
                CompositeMetadata cm = holder.getCompositeMetadata(name);
                _saxMetaData(cm, isLocked, isModifiable);
                break;
            case BINARY:
                String mimeType = holder.getBinaryMetadata(name).getMimeType();
                AttributesImpl attrsimpl = new AttributesImpl();
                attrsimpl.addCDATAAttribute("mime-type", mimeType);
                XMLUtils.createElement(contentHandler, "value", attrsimpl);
                break;
            case RICHTEXT:
                RichText rt = holder.getRichText(name);
                _saxRichText(rt);
                break;
            case OBJECT_COLLECTION:
                // TODO
                break;
            default:
                textValue = "";
                XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                break;
        }

    }

    private void _saxSimpleValue(CompositeMetadata holder, String name, MetadataType type, boolean isLocked, boolean isModifiable) throws RepositoryException, SAXException
    {
        Attributes attrs = XMLUtils.EMPTY_ATTRIBUTES;
        String textValue = null;

        switch (type)
        {
            case DATE:
                XMLUtils.createElement(contentHandler, "value", attrs, DateUtils.dateToString(holder.getDate(name)));
                break;
            case STRING:
                textValue = holder.getString(name);
                XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                break;
            case LONG:
                textValue = Long.toString(holder.getLong(name));
                XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                break;
            case DOUBLE:
                textValue = Double.toString(holder.getDouble(name));
                XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                break;
            case BOOLEAN:
                textValue = Boolean.toString(holder.getBoolean(name));
                XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                break;
            case COMPOSITE:
                CompositeMetadata cm = holder.getCompositeMetadata(name);
                _saxMetaData(cm, isLocked, isModifiable);
                break;
            case BINARY:
                String mimeType = holder.getBinaryMetadata(name).getMimeType();
                AttributesImpl attrsimpl = new AttributesImpl();
                attrsimpl.addCDATAAttribute("mime-type", mimeType);
                XMLUtils.createElement(contentHandler, "value", attrsimpl);
                break;
            case RICHTEXT:
                RichText rt = holder.getRichText(name);
                _saxRichText(rt);
                break;
            case OBJECT_COLLECTION:
                // TODO
                break;
            default:
                textValue = "";
                XMLUtils.createElement(contentHandler, "value", attrs, textValue);
                break;
        }
    }

    private void _saxRichText(RichText rt) throws RepositoryException, SAXException
    {
        //content
        AttributesImpl contentAttributes = new AttributesImpl();
        contentAttributes.addCDATAAttribute("name", "data");
        contentAttributes.addCDATAAttribute("type", "RICHTEXT-CONTENT");
        contentAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", contentAttributes);  
        
        AttributesImpl attrsimplRT = new AttributesImpl();
        attrsimplRT.addCDATAAttribute("mime-type", rt.getMimeType());
        XMLUtils.createElement(contentHandler, "value", attrsimplRT);
        
        XMLUtils.endElement(contentHandler, "metadata");
        
        //encoding
        if (rt.getEncoding() != null)
        {
            AttributesImpl encodingAttributes = new AttributesImpl();
            encodingAttributes.addCDATAAttribute("name", "encoding");
            encodingAttributes.addCDATAAttribute("type", "STRING");
            encodingAttributes.addCDATAAttribute("isMultiple", "false");
            XMLUtils.startElement(contentHandler, "metadata", encodingAttributes);
            
            XMLUtils.createElement(contentHandler, "value", XMLUtils.EMPTY_ATTRIBUTES, rt.getEncoding());
            
            XMLUtils.endElement(contentHandler, "metadata");
        }
        
        //mime-type
        AttributesImpl mimeAttributes = new AttributesImpl();
        mimeAttributes.addCDATAAttribute("name", "mime-type");
        mimeAttributes.addCDATAAttribute("type", "STRING");
        mimeAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", mimeAttributes);
        
        XMLUtils.createElement(contentHandler, "value", XMLUtils.EMPTY_ATTRIBUTES, rt.getMimeType());
        
        XMLUtils.endElement(contentHandler, "metadata");
        
        //last modified
        AttributesImpl lastmodAttributes = new AttributesImpl();
        lastmodAttributes.addCDATAAttribute("name", "last-modified-date");
        lastmodAttributes.addCDATAAttribute("type", "DATE");
        lastmodAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", lastmodAttributes);
        
        XMLUtils.createElement(contentHandler, "value", XMLUtils.EMPTY_ATTRIBUTES, DateUtils.dateToString(rt.getLastModified()));
        
        XMLUtils.endElement(contentHandler, "metadata");
        
        //length
        AttributesImpl lengthAttributes = new AttributesImpl();
        lengthAttributes.addCDATAAttribute("name", "length");
        lengthAttributes.addCDATAAttribute("type", "LONG");
        lengthAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", lengthAttributes);
        
        XMLUtils.createElement(contentHandler, "value", XMLUtils.EMPTY_ATTRIBUTES, Long.toString(rt.getLength()));
        
        XMLUtils.endElement(contentHandler, "metadata");
        
        //sub folders
        _saxFolder(rt.getAdditionalDataFolder());
    }

    private void _saxFolder(Folder folder) throws RepositoryException, SAXException
    {
        AttributesImpl folderAttributes = new AttributesImpl();
        folderAttributes.addCDATAAttribute("name", folder.getName());
        folderAttributes.addCDATAAttribute("type", "FOLDER");
        folderAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", folderAttributes);

        Collection<? extends Folder> folders = folder.getFolders();
        Iterator itFolders = folders.iterator();
        while (itFolders.hasNext())
        {
            Folder subfold = (Folder) itFolders.next();
            _saxFolder(subfold);
        }

        Collection<? extends File> files = folder.getFiles();
        Iterator itFiles = files.iterator();
        while (itFiles.hasNext())
        {
            File file = (File) itFiles.next();
            _saxFile(file);
        }

        XMLUtils.endElement(contentHandler, "metadata");
    }



    private void _saxFile(File file) throws SAXException
    {
        AttributesImpl fileAttributes = new AttributesImpl();
        fileAttributes.addCDATAAttribute("name", file.getName());
        fileAttributes.addCDATAAttribute("type", "FILE");
        fileAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", fileAttributes);

        //content
        AttributesImpl contentAttributes = new AttributesImpl();
        contentAttributes.addCDATAAttribute("name", "data");
        contentAttributes.addCDATAAttribute("type", "FILE-CONTENT");
        contentAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", contentAttributes);  

        AttributesImpl attrsimplF = new AttributesImpl();
        attrsimplF.addCDATAAttribute("mime-type", file.getResource().getMimeType());
        XMLUtils.createElement(contentHandler, "value", attrsimplF);

        XMLUtils.endElement(contentHandler, "metadata");

        //encoding
        if (file.getResource().getEncoding() != null)
        {
            AttributesImpl encodingAttributes = new AttributesImpl();
            encodingAttributes.addCDATAAttribute("name", "encoding");
            encodingAttributes.addCDATAAttribute("type", "STRING");
            encodingAttributes.addCDATAAttribute("isMultiple", "false");
            XMLUtils.startElement(contentHandler, "metadata", encodingAttributes);

            XMLUtils.createElement(contentHandler, "value", XMLUtils.EMPTY_ATTRIBUTES, file.getResource().getEncoding());

            XMLUtils.endElement(contentHandler, "metadata");
        }

        //mime-type
        AttributesImpl mimeAttributes = new AttributesImpl();
        mimeAttributes.addCDATAAttribute("name", "mime-type");
        mimeAttributes.addCDATAAttribute("type", "STRING");
        mimeAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", mimeAttributes);

        XMLUtils.createElement(contentHandler, "value", XMLUtils.EMPTY_ATTRIBUTES, file.getResource().getMimeType());

        XMLUtils.endElement(contentHandler, "metadata");

        //last modified
        AttributesImpl lastmodAttributes = new AttributesImpl();
        lastmodAttributes.addCDATAAttribute("name", "last-modified-date");
        lastmodAttributes.addCDATAAttribute("type", "DATE");
        lastmodAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", lastmodAttributes);

        XMLUtils.createElement(contentHandler, "value", XMLUtils.EMPTY_ATTRIBUTES, DateUtils.dateToString(file.getResource().getLastModified()));

        XMLUtils.endElement(contentHandler, "metadata");

        //length
        AttributesImpl lengthAttributes = new AttributesImpl();
        lengthAttributes.addCDATAAttribute("name", "length");
        lengthAttributes.addCDATAAttribute("type", "LONG");
        lengthAttributes.addCDATAAttribute("isMultiple", "false");
        XMLUtils.startElement(contentHandler, "metadata", lengthAttributes);

        XMLUtils.createElement(contentHandler, "value", XMLUtils.EMPTY_ATTRIBUTES, Long.toString(file.getResource().getLength()));

        XMLUtils.endElement(contentHandler, "metadata");

        XMLUtils.endElement(contentHandler, "metadata");
    }

}
