/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.workspace;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.servlet.multipart.PartOnDisk;
import org.apache.commons.lang3.exception.ExceptionUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.metadata.CompositeMetadata.MetadataType;
import org.ametys.plugins.repository.metadata.ModifiableCompositeMetadata;
import org.ametys.plugins.repository.metadata.ModifiableFile;
import org.ametys.plugins.repository.metadata.ModifiableFolder;
import org.ametys.plugins.repository.metadata.ModifiableMetadataAwareAmetysObject;
import org.ametys.plugins.repository.metadata.ModifiableRichText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Set value for a metadata
 */
public class SetMetadataAction extends ServiceableAction
{
    static SimpleDateFormat _DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
    private AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver sourceResolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String id = null; 
        String name = null;
        String compositePath = null;
        String richtextPath = null;
        boolean multiple;
        
        String type = null;
        String fileFieldId = null;
    
        try
        {
            id = parameters.getParameter("id");
            name = parameters.getParameter("name");
            compositePath = parameters.getParameter("cPath");
            richtextPath = parameters.getParameter("rtPath");
            type = parameters.getParameter("type");
            fileFieldId = parameters.getParameter("filefieldid");
            multiple = parameters.getParameterAsBoolean("multiple", false);
        }
        catch (ParameterException e)
        {
            throw new ProcessingException("Missing at least one parameter", e);
        }
        
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Setting metadata '" + name + "' of the AmetysObject of id '" + id + "'");
        }
        
        AmetysObject ao = null;
        
        if (id.equals("/"))
        {
            ao = _resolver.resolveByPath("/");
        }
        else
        {
            ao = _resolver.resolveById(id);
        }

        String[] values = request.getParameterValues("value");
        
        Map<String, Object> result = new HashMap<>();
        
        try
        {
            ModifiableCompositeMetadata holder = _getHolderInComposite((ModifiableMetadataAwareAmetysObject) ao, compositePath);
            
            //changing value of a metadata in a composite
            if (richtextPath.isEmpty())
            {
                _setMetadataOutsideRT(holder, type, name, values, multiple);          
            }
            //changing a value inside a richtext
            else if (richtextPath.split("/").length == 1)
            {
                _setMetadataInsideRT(request, holder, name, values[0], fileFieldId, richtextPath);        
            }
            else //TODO this part is not tested !!
            {
                _setMetadataInsideRTDataFolder(request, holder, name, values[0], fileFieldId, richtextPath);
            }
    
            ((ModifiableMetadataAwareAmetysObject) ao).saveChanges();
            
            result.put("success", true);
        }
        catch (Exception e)
        {
            getLogger().error("Unable to set value " + values[0] + " for metadata '" + name + "'", e);
            result.put("success", false);
            
            Throwable rootCause = ExceptionUtils.getRootCause(e);
            
            if (rootCause instanceof ParseException)
            {
                Map<String, I18nizableTextParameter> params = Collections.singletonMap("error", new I18nizableText(e.getMessage()));
                result.put("message", new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_CREATE_PROP_ERROR_DATE", params));
            }
            else
            {
                result.put("message", e.toString());
            }
        }

        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    private ModifiableCompositeMetadata _getHolderInComposite(ModifiableMetadataAwareAmetysObject ao, String compositePath)
    {
        ModifiableCompositeMetadata holder = ao.getMetadataHolder();

        // deep search in composite metadata
        if (!compositePath.isEmpty())
        {
            String[] tokens = compositePath.split("/");        
            for (int i = 0; i < tokens.length; i++)
            {
                holder = holder.getCompositeMetadata(tokens[i]);
            }
        }
        
        return holder;
    }
    
    private void _setMetadataOutsideRT(ModifiableCompositeMetadata holder, String type, String name, String[] values, boolean multiple) throws Exception
    {
        //switch according to type
        //convert value to correct type if not string and set new value          
        MetadataType mdType = MetadataType.valueOf(type.toUpperCase());
        switch (mdType)
        {
            case DATE:
                if (multiple)
                {
                    Date[] dateValues = Arrays.stream(values).map(LambdaUtils.wrap(_DATE_FORMAT::parse)).toArray(Date[]::new);
                    holder.setMetadata(name, dateValues);
                }
                else
                {
                    Date dateVal = _DATE_FORMAT.parse(values[0]);
                    holder.setMetadata(name, dateVal);
                }
                break;
            case STRING:
                if (multiple)
                {
                    holder.setMetadata(name, values);
                }
                else
                {
                    holder.setMetadata(name, values[0]);
                }
                break;
            case LONG:
                if (multiple)
                {
                    long[] longValues = Arrays.stream(values).mapToLong(Long::parseLong).toArray();
                    holder.setMetadata(name, longValues);
                }
                else
                {
                    Long longVal = Long.parseLong(values[0]);
                    holder.setMetadata(name, longVal);
                }
                break;
            case DOUBLE:
                if (multiple)
                {
                    double[] doubleValues = Arrays.stream(values).mapToDouble(Double::parseDouble).toArray();
                    holder.setMetadata(name, doubleValues);
                }
                else
                {
                    Double doubleVal = Double.parseDouble(values[0]);
                    holder.setMetadata(name, doubleVal);
                }
                
                break;
            case BOOLEAN:
                Boolean booleanVal = Boolean.parseBoolean(values[0]);
                holder.setMetadata(name, booleanVal);
                break;
            case COMPOSITE:

                break;
            case BINARY:

                break;
            case RICHTEXT:

                break;
            case OBJECT_COLLECTION:
                // TODO?
                break;
            default:
                break;
        }
    }
    
    private void _setMetadataInsideRT(Request request, ModifiableCompositeMetadata holder, String name, String value, String fileFieldId, String richtextPath) throws Exception
    {
        String[] rtTokens = richtextPath.split("/");
        ModifiableRichText rt = holder.getRichText(rtTokens[0]);
        
        // update correct property
        if (name.equals("mime-type"))
        {
            rt.setMimeType(value);
        }
        else if (name.equals("last-modified-date"))
        {
            Date dateVal = _DATE_FORMAT.parse(value);
            rt.setLastModified(dateVal);
        }
        else if (name.equals("encoding"))
        {
            rt.setEncoding(value);
        }
        else if (name.equals("data"))
        {
            PartOnDisk uploadedFilePart = (PartOnDisk) request.get(fileFieldId);
            File uploadedFile = (uploadedFilePart != null) ? uploadedFilePart.getFile() : null;
            
            if (uploadedFile != null)
            {
                InputStream is = new FileInputStream(uploadedFile);
                rt.setInputStream(is);
            }
        }
        else if (name.equals("length"))
        {
            //not possible to change length
        }
    }
    
    private void _setMetadataInsideRTDataFolder(Request request, ModifiableCompositeMetadata holder, String name, String value, String fileFieldId, String richtextPath) throws Exception
    {
        String[] rtTokens = richtextPath.split("/");
        ModifiableRichText rt = holder.getRichText(rtTokens[0]);
        ModifiableFolder folder = rt.getAdditionalDataFolder();
        
        for (int i = 2; i < rtTokens.length - 1; i++)
        {
            folder = folder.getFolder(rtTokens[i]);
        }
        
        ModifiableFile file = folder.getFile(rtTokens[rtTokens.length - 1]);
        
        // update correct property
        if (name.equals("mime-type"))
        {
            file.getResource().setMimeType(value);
        }
        else if (name.equals("last-modified-date"))
        {
            Date dateVal = _DATE_FORMAT.parse(value);
            file.getResource().setLastModified(dateVal);
        }
        else if (name.equals("encoding"))
        {
            file.getResource().setEncoding(value);
        }
        else if (name.equals("data"))
        {
            PartOnDisk uploadedFilePart = (PartOnDisk) request.get(fileFieldId);
            File uploadedFile = (uploadedFilePart != null) ? uploadedFilePart.getFile() : null;
            InputStream is = new FileInputStream(uploadedFile);
            file.getResource().setInputStream(is);
        }
        else if (name.equals("length"))
        {
            //not possible to change length
        }
    }
}
