/*
 * Decompiled with CFR 0.152.
 */
package org.ametys.core.util.path;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileExistsException;

public final class PathUtils {
    private PathUtils() {
    }

    public static void copyDirectory(Path srcDir, Path destDir) throws IOException {
        PathUtils.copyDirectory(srcDir, destDir, true);
    }

    public static void copyDirectory(Path srcDir, Path destDir, boolean preserveFileDate) throws IOException {
        PathUtils.copyDirectory(srcDir, destDir, null, preserveFileDate);
    }

    public static void copyDirectory(Path srcDir, Path destDir, Predicate<Path> filter) throws IOException {
        PathUtils.copyDirectory(srcDir, destDir, filter, true);
    }

    public static void copyDirectory(Path srcDir, Path destDir, Predicate<Path> filter, boolean preserveFileDate) throws IOException {
        PathUtils.checkFileRequirements(srcDir, destDir);
        if (!Files.isDirectory(srcDir, new LinkOption[0])) {
            throw new IOException("Source '" + String.valueOf(srcDir) + "' exists but is not a directory");
        }
        if (srcDir.toAbsolutePath().equals(destDir.toAbsolutePath())) {
            throw new IOException("Source '" + String.valueOf(srcDir) + "' and destination '" + String.valueOf(destDir) + "' are the same");
        }
        ArrayList<Path> exclusionList = null;
        if (destDir.getFileSystem() == srcDir.getFileSystem() && destDir.toAbsolutePath().startsWith(srcDir.toAbsolutePath())) {
            List srcFiles;
            try (Stream<Path> walk = Files.list(srcDir);){
                srcFiles = filter != null ? walk.filter(filter).collect(Collectors.toList()) : walk.collect(Collectors.toList());
            }
            if (!srcFiles.isEmpty()) {
                exclusionList = new ArrayList<Path>(srcFiles.size());
                for (Path srcFile : srcFiles) {
                    Path copiedFile = destDir.resolve(srcFile.getFileName());
                    exclusionList.add(copiedFile.toAbsolutePath());
                }
            }
        }
        PathUtils.doCopyDirectory(srcDir, destDir, filter, preserveFileDate, exclusionList);
    }

    private static void checkFileRequirements(Path src, Path dest) throws FileNotFoundException {
        if (src == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (dest == null) {
            throw new NullPointerException("Destination must not be null");
        }
        if (!Files.exists(src, new LinkOption[0])) {
            throw new FileNotFoundException("Source '" + String.valueOf(src) + "' does not exist");
        }
    }

    private static void doCopyDirectory(Path srcDir, Path destDir, Predicate<Path> filter, boolean preserveFileDate, List<Path> exclusionList) throws IOException {
        List srcFiles;
        try (Stream<Path> walk = Files.list(srcDir);){
            srcFiles = filter != null ? walk.filter(filter).collect(Collectors.toList()) : walk.collect(Collectors.toList());
        }
        if (Files.exists(destDir, new LinkOption[0])) {
            if (!Files.isDirectory(destDir, new LinkOption[0])) {
                throw new IOException("Destination '" + String.valueOf(destDir) + "' exists but is not a directory");
            }
        } else {
            Path createdDirectories = Files.createDirectories(destDir, new FileAttribute[0]);
            if (!Files.isDirectory(createdDirectories, new LinkOption[0])) {
                throw new IOException("Destination '" + String.valueOf(destDir) + "' directory cannot be created");
            }
        }
        if (!Files.isWritable(destDir)) {
            throw new IOException("Destination '" + String.valueOf(destDir) + "' cannot be written to");
        }
        for (Path srcFile : srcFiles) {
            Path dstFile = destDir.resolve(srcFile.getFileName().toString());
            if (exclusionList != null && exclusionList.contains(srcFile.toAbsolutePath())) continue;
            if (Files.isDirectory(srcFile, new LinkOption[0])) {
                PathUtils.doCopyDirectory(srcFile, dstFile, filter, preserveFileDate, exclusionList);
                continue;
            }
            PathUtils.doCopyFile(srcFile, dstFile, preserveFileDate);
        }
        if (preserveFileDate) {
            Files.setLastModifiedTime(destDir, Files.getLastModifiedTime(srcDir, new LinkOption[0]));
        }
    }

    private static void doCopyFile(Path srcFile, Path destFile, boolean preserveFileDate) throws IOException {
        if (Files.exists(destFile, new LinkOption[0]) && Files.isDirectory(destFile, new LinkOption[0])) {
            throw new IOException("Destination '" + String.valueOf(destFile) + "' exists but is a directory");
        }
        Files.copy(srcFile, destFile, StandardCopyOption.REPLACE_EXISTING);
        long srcLen = Files.size(srcFile);
        long dstLen = Files.size(destFile);
        if (srcLen != dstLen) {
            throw new IOException("Failed to copy full contents from '" + String.valueOf(srcFile) + "' to '" + String.valueOf(destFile) + "' Expected length: " + srcLen + " Actual: " + dstLen);
        }
        if (preserveFileDate) {
            Files.setLastModifiedTime(destFile, Files.getLastModifiedTime(srcFile, new LinkOption[0]));
        }
    }

    public static void moveFile(Path srcFile, Path destFile) throws IOException {
        block8: {
            if (srcFile == null) {
                throw new NullPointerException("Source must not be null");
            }
            if (destFile == null) {
                throw new NullPointerException("Destination must not be null");
            }
            if (!Files.exists(srcFile, new LinkOption[0])) {
                throw new FileNotFoundException("Source '" + String.valueOf(srcFile) + "' does not exist");
            }
            if (Files.isDirectory(srcFile, new LinkOption[0])) {
                throw new IOException("Source '" + String.valueOf(srcFile) + "' is a directory");
            }
            if (Files.exists(destFile, new LinkOption[0])) {
                throw new FileExistsException("Destination '" + String.valueOf(destFile) + "' already exists");
            }
            if (Files.isDirectory(destFile, new LinkOption[0])) {
                throw new IOException("Destination '" + String.valueOf(destFile) + "' is a directory");
            }
            try {
                Files.move(srcFile, destFile, new CopyOption[0]);
            }
            catch (IOException e) {
                PathUtils.copyFile(srcFile, destFile);
                if (Files.deleteIfExists(srcFile)) break block8;
                PathUtils.deleteQuietly(destFile);
                throw new IOException("Failed to delete original file '" + String.valueOf(srcFile) + "' after copy to '" + String.valueOf(destFile) + "'");
            }
        }
    }

    public static void moveToDirectory(Path src, Path destDir, boolean createDestDir) throws IOException {
        if (src == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (destDir == null) {
            throw new NullPointerException("Destination must not be null");
        }
        if (!Files.exists(src, new LinkOption[0])) {
            throw new FileNotFoundException("Source '" + String.valueOf(src) + "' does not exist");
        }
        if (Files.isDirectory(src, new LinkOption[0])) {
            PathUtils.moveDirectoryToDirectory(src, destDir, createDestDir);
        } else {
            PathUtils.moveFileToDirectory(src, destDir, createDestDir);
        }
    }

    public static void moveFileToDirectory(Path srcFile, Path destDir, boolean createDestDir) throws IOException {
        if (srcFile == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (destDir == null) {
            throw new NullPointerException("Destination directory must not be null");
        }
        if (!Files.exists(destDir, new LinkOption[0]) && createDestDir) {
            Files.createDirectories(destDir, new FileAttribute[0]);
        }
        if (!Files.exists(destDir, new LinkOption[0])) {
            throw new FileNotFoundException("Destination directory '" + String.valueOf(destDir) + "' does not exist [createDestDir=" + createDestDir + "]");
        }
        if (!Files.isDirectory(destDir, new LinkOption[0])) {
            throw new IOException("Destination '" + String.valueOf(destDir) + "' is not a directory");
        }
        PathUtils.moveFile(srcFile, destDir.resolve(srcFile.getFileName().toString()));
    }

    public static void moveDirectoryToDirectory(Path src, Path destDir, boolean createDestDir) throws IOException {
        if (src == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (destDir == null) {
            throw new NullPointerException("Destination directory must not be null");
        }
        if (!Files.exists(destDir, new LinkOption[0]) && createDestDir) {
            Files.createDirectories(destDir, new FileAttribute[0]);
        }
        if (!Files.exists(destDir, new LinkOption[0])) {
            throw new FileNotFoundException("Destination directory '" + String.valueOf(destDir) + "' does not exist [createDestDir=" + createDestDir + "]");
        }
        if (!Files.isDirectory(destDir, new LinkOption[0])) {
            throw new IOException("Destination '" + String.valueOf(destDir) + "' is not a directory");
        }
        PathUtils.moveDirectory(src, destDir.resolve(src.getFileName().toString()));
    }

    public static void moveDirectory(Path srcDir, Path destDir) throws IOException {
        block8: {
            if (srcDir == null) {
                throw new NullPointerException("Source must not be null");
            }
            if (destDir == null) {
                throw new NullPointerException("Destination must not be null");
            }
            if (!Files.exists(srcDir, new LinkOption[0])) {
                throw new FileNotFoundException("Source '" + String.valueOf(srcDir) + "' does not exist");
            }
            if (!Files.isDirectory(srcDir, new LinkOption[0])) {
                throw new IOException("Source '" + String.valueOf(srcDir) + "' is not a directory");
            }
            if (Files.exists(destDir, new LinkOption[0])) {
                throw new FileExistsException("Destination '" + String.valueOf(destDir) + "' already exists");
            }
            try {
                Files.move(srcDir, destDir, new CopyOption[0]);
            }
            catch (IOException e) {
                if (destDir.toAbsolutePath().toString().startsWith(srcDir.toAbsolutePath().toString() + File.separator)) {
                    throw new IOException("Cannot move directory: " + String.valueOf(srcDir) + " to a subdirectory of itself: " + String.valueOf(destDir));
                }
                PathUtils.copyDirectory(srcDir, destDir);
                PathUtils.deleteDirectory(srcDir);
                if (!Files.exists(srcDir, new LinkOption[0])) break block8;
                throw new IOException("Failed to delete original directory '" + String.valueOf(srcDir) + "' after copy to '" + String.valueOf(destDir) + "'");
            }
        }
    }

    public static boolean deleteQuietly(Path file) {
        if (file == null) {
            return false;
        }
        try {
            if (Files.isDirectory(file, new LinkOption[0])) {
                PathUtils.cleanDirectory(file);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        try {
            return Files.deleteIfExists(file);
        }
        catch (Exception ignored) {
            return false;
        }
    }

    public static void cleanDirectory(Path directory) throws IOException {
        Path[] files = PathUtils.verifiedListFiles(directory);
        IOException exception = null;
        for (Path file : files) {
            try {
                PathUtils.forceDelete(file);
            }
            catch (IOException ioe) {
                exception = ioe;
            }
        }
        if (null != exception) {
            throw exception;
        }
    }

    private static Path[] verifiedListFiles(Path directory) throws IOException {
        if (!Files.exists(directory, new LinkOption[0])) {
            String message = String.valueOf(directory) + " does not exist";
            throw new IllegalArgumentException(message);
        }
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            String message = String.valueOf(directory) + " is not a directory";
            throw new IllegalArgumentException(message);
        }
        try (Stream<Path> s = Files.list(directory);){
            Path[] pathArray = (Path[])s.toArray(Path[]::new);
            return pathArray;
        }
    }

    public static void forceDelete(Path file) throws IOException {
        if (Files.isDirectory(file, new LinkOption[0])) {
            PathUtils.deleteDirectory(file);
        } else {
            boolean filePresent = Files.exists(file, new LinkOption[0]);
            if (!Files.deleteIfExists(file)) {
                if (!filePresent) {
                    throw new FileNotFoundException("File does not exist: " + String.valueOf(file));
                }
                String message = "Unable to delete file: " + String.valueOf(file);
                throw new IOException(message);
            }
        }
    }

    public static void deleteDirectory(Path directory) throws IOException {
        if (!Files.exists(directory, new LinkOption[0])) {
            return;
        }
        if (!PathUtils.isSymlink(directory)) {
            PathUtils.cleanDirectory(directory);
        }
        if (!Files.deleteIfExists(directory)) {
            String message = "Unable to delete directory " + String.valueOf(directory) + ".";
            throw new IOException(message);
        }
    }

    public static boolean isSymlink(Path file) throws IOException {
        if (file == null) {
            throw new NullPointerException("File must not be null");
        }
        return Files.isSymbolicLink(file);
    }

    public static void copyFile(Path srcFile, Path destFile) throws IOException {
        PathUtils.copyFile(srcFile, destFile, true);
    }

    public static void copyFile(Path srcFile, Path destFile, boolean preserveFileDate) throws IOException {
        PathUtils.checkFileRequirements(srcFile, destFile);
        if (Files.isDirectory(srcFile, new LinkOption[0])) {
            throw new IOException("Source '" + String.valueOf(srcFile) + "' exists but is a directory");
        }
        if (srcFile.toAbsolutePath().equals(destFile.toAbsolutePath())) {
            throw new IOException("Source '" + String.valueOf(srcFile) + "' and destination '" + String.valueOf(destFile) + "' are the same");
        }
        Path parentFile = destFile.getParent();
        if (parentFile != null) {
            Files.createDirectories(parentFile, new FileAttribute[0]);
            if (!Files.isDirectory(parentFile, new LinkOption[0])) {
                throw new IOException("Destination '" + String.valueOf(parentFile) + "' directory cannot be created");
            }
        }
        if (Files.exists(destFile, new LinkOption[0]) && !Files.isWritable(destFile)) {
            throw new IOException("Destination '" + String.valueOf(destFile) + "' exists but is read-only");
        }
        PathUtils.doCopyFile(srcFile, destFile, preserveFileDate);
    }
}

