/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class to handle maintenance mode
 */
Ext.define('Ametys.plugins.admin.maintenance.Maintenance', {
    singleton: true,

    /**
     * Activate or deactivate the maintenance mode
     * @param controller The controller calling this function
     */
    act: function (controller) 
    {
        if (controller.isPressed())
        {
            // Remove maintenance => no message
            controller.serverCall("goToNormalMode", [], this._afterChangeGoToNormalMode, { arguments: controller })
        }
        else
        {
            // Set maintenance => ask message
            this._ask(controller);
        }
    },
    
    /**
     * @private
     * Ask message
     */
    _ask: function(controller)
    {
        var dialog = Ext.create('Ametys.window.DialogBox', {
            title: "{{i18n PLUGINS_ADMIN_MAINTENANCE_COMMENT_DIALOG_TITLE}}",
            iconCls: 'ametysicon-object-hammer-wrench',
            cls: 'text-dialog',
            
            width: 450,
            scrollable: false,
            
            items: [{
                        cls: 'ametys',
                        xtype: 'component',
                        html: "{{i18n PLUGINS_ADMIN_MAINTENANCE_COMMENT_DIALOG_TEXT}}",
                        width: '100%'
                    }, 
                    {
                        cls: 'ametys',
                        xtype: 'textareafield',
                        name: 'comment',
                        id: 'maintenance-comment',
                        fieldLabel: "{{i18n PLUGINS_ADMIN_MAINTENANCE_COMMENT_DIALOG_COMMENT}}",
                        width: '100%',
                        labelAlign: 'top',
                        height: 140
                    }
            ],
            
            defaultFocus: 'maintenance-comment',
            closeAction: 'destroy',
            buttons : [{
                text :"{{i18n PLUGINS_ADMIN_MAINTENANCE_COMMENT_DIALOG_OK}}",
                handler : Ext.bind(function() {
                    controller.serverCall("goToMaintenanceMode", [dialog.getComponent("maintenance-comment").getValue()], this._afterChangeGoToMaintenanceMode, { arguments: [controller, dialog] })
                }, this)
            }, {
                text :"{{i18n PLUGINS_ADMIN_MAINTENANCE_COMMENT_DIALOG_CANCEL}}",
                handler: Ext.bind(function() {dialog.close();}, this) 
            }]
        });
        dialog.show();
    },
    
    /**
     * @private
     * Callback after changing maintenance to normal mode
     * @param {Object} v void
     * @param controller The controller calling this function
     */
    _afterChangeGoToNormalMode: function(v, controller)
    {
        controller.toggle(false);
        controller.setDescription(controller.getInitialConfig("forced-off-description"));
        controller.setIconDecorator(controller.getInitialConfig("off-icon-decorator"));
        controller.setIconDecoratorType(controller.getInitialConfig("off-icon-decorator-type"));
        
        Ext.create('Ametys.message.Message', {
            type: Ametys.message.Message.MODIFIED,
            targets: [{
                id: 'maintenance',
                parameters: {
                    state: 'deactivated'
                }
            }]
        });
        
        Ametys.plugins.admin.maintenance.Maintenance.forceMaintenanceAnnounceUpdate();
    },
    
    
    /**
     * @private
     * Callback after changing maintenance to normal mode
     * @param {Object} v void
     * @param Object[] opt Transmistted args
     */
    _afterChangeGoToMaintenanceMode: function(v, opt)
    {
        var controller = opt[0]
        var dialog = opt[1];
        
        dialog.close();
        
        controller.toggle(true);
        controller.setDescription(controller.getInitialConfig("forced-on-description"));
        controller.setIconDecorator(controller.getInitialConfig("on-icon-decorator"));
        controller.setIconDecoratorType(controller.getInitialConfig("on-icon-decorator-type"));
        
        Ext.create('Ametys.message.Message', {
            type: Ametys.message.Message.MODIFIED,
            targets: [{
                id: 'maintenance',
                parameters: {
                    state: 'activated'
                }
            }]
        });

        Ametys.plugins.admin.maintenance.Maintenance.forceMaintenanceAnnounceUpdate();
    },
    
    /**
     * @private
     * This method send a fake request to server to force the update of system announcement on the local HMI
     */
    forceMaintenanceAnnounceUpdate: function()
    {
        // We need to send a server request (whatever the request) to force the HMI update
        Ametys.data.ServerComm.callMethod({
            role: "org.ametys.runtime.plugins.admin.system.SystemHelper",
            methodName: "isSystemAnnouncementAvailable",
            parameters: []
        });
    },
});

