/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class controls ribbon buttons managing the state of the system announcement message
 * @private
 */
Ext.define('Ametys.plugins.admin.system.SystemAnnouncementActivationController', {
	extend: 'Ametys.ribbon.element.ui.ButtonController',
	
/**
     * @cfg {Boolean} [available=false] True if the system announcement is available
     */
    /**
     * @cfg {String} [announcement-on-icon-decorator] The CSS class for decorator when the system announcement is on
     */
    /**
     * @cfg {String} [announcement-on-icon-decorator-type] The decorator type when the system announcement is on
     */
    /**
     * @cfg {String} [announcement-off-icon-decorator] The CSS class for decorator when the system announcement is off
     */
    /**
     * @cfg {String} [announcement-off-icon-decorator-type] The decorator type when the system announcement is off
     */
    
    /**
     * @cfg {String} [announcement-on-description] The description when when the system announcement is on
     */
    /**
     * @cfg {String} [announcement-off-description] The description when when the system announcement is off
     */
    
    /**
     * @property {String} _onIconDecorator See #cfg-announcement-on-icon-decorator
     * @private
     */
    /**
     * @property {String} _onIconDecoratorType See #cfg-announcement-on-icon-decorator-type
     * @private
     */
    /**
     * @property {String} _offIconDecorator See #cfg-announcement-off-icon-decorator
     * @private
     */
    /**
     * @property {String} _offIconDecoratorType See #cfg-announcement-off-icon-decorator-type
     * @private
     */
	
	constructor: function(config)
	{
		this.callParent(arguments);
        
        this._offIconDecorator = this.getInitialConfig("announcement-off-icon-decorator");
        this._offIconDecoratorType = this.getInitialConfig("announcement-off-icon-decorator-type");
        this._onIconDecorator = this.getInitialConfig("announcement-on-icon-decorator");
        this._onIconDecoratorType = this.getInitialConfig("announcement-on-icon-decorator-type");
        this._scheduledIconDecorator = this.getInitialConfig("announcement-scheduled-icon-decorator");
        this._scheduledIconDecoratorType = this.getInitialConfig("announcement-scheduled-icon-decorator-type");
        
        this.startDate = config['start-date'];
        this.endDate = config['end-date'];
        this.state = config['state'];
        this._updateDecorator();

		Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
	},

	/**
	 * Listener when the toggle state of the button has been initialized/modified
	 * Will update the state of the button
	 * @param {Ametys.message.Message} message The modified message.
	 * @protected
	 */
	_onModified: function (message)
	{
		var target = message.getTarget('system-announcement');
		if (target != null)
		{
            this.startDate = target.getParameters().startDate;
            this.endDate = target.getParameters().endDate;
            this.state = target.getParameters().state;
            this._updateDecorator();
		}
	},
	
	_updateDecorator()
	{
        var decorator;
        var type;
        var description;
        var toggle;
        switch (this.state)
        {
            case "scheduled":
                decorator = this._scheduledIconDecorator;
                // description depends solely on existance of date
                if (this.startDate == null)
                {
                    if (this.endDate == null)
                    {
                        decorator = this._onIconDecorator;
                        description = this.getInitialConfig("announcement-on-description") || null;
                    }
                    else
                    {
                        description = Ext.String.format(this.getInitialConfig("announcement-scheduled-until-description"), this._formatDate(this.endDate)) || null;
                    }
                }
                else
                {
                    if (this.endDate == null)
                    {
                        description = Ext.String.format(this.getInitialConfig("announcement-scheduled-after-description"), this._formatDate(this.startDate)) || null;
                    }
                    else
                    {
                        description = Ext.String.format(this.getInitialConfig("announcement-scheduled-between-description"), this._formatDate(this.startDate), this._formatDate(this.endDate)) || null;
                    }
                }
                
                // toggle and type depends of comparaison with current date
                var now = new Date();
                if (this.startDate == null || new Date(this.startDate) < now) // after start date
                {
                    if (this.endDate == null || now < new Date(this.endDate)) // before end date
                    {
                        toggle = true;
                        type = this._onIconDecoratorType;
                    }
                    else
                    {
                        toggle = false;
                        type = this._offIconDecoratorType;
                    }
                }
                else
                {
                    toggle = false;
                    type = this._scheduledIconDecoratorType;
                }
                break;
            case "on":
                toggle = true;
                decorator = this._onIconDecorator;
                type = this._onIconDecoratorType;
                description = this.getInitialConfig("announcement-on-description") || null;
                break;
            default:
                toggle = false;
                decorator = this._offIconDecorator;
                type = this._offIconDecoratorType;
                description = this.getInitialConfig("announcement-off-description") || null;
        }
        
        this.toggle(toggle);
        this.setIconDecorator(decorator);
        this.setIconDecoratorType(type);
        if (toggle)
        {
            description += this.getInitialConfig("announcement-disable-description");
        }
        else
        {
            description += this.getInitialConfig("announcement-enable-description");
        }
        this.setAdditionalDescription (description);
    },
    
    _formatDate:function(strDate)
    {
        var date = Ext.Date.parse(strDate, Ext.Date.patterns.ISO8601DateTime);
        return Ext.Date.format(date, Ext.Date.patterns.FriendlyDateTime)
    }
});
