/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Actions for unknown users
 * @private
 */
Ext.define('Ametys.plugins.admin.user.UnknownUsersActions', {
    singleton: true,
    
    /**
     * Activate the personal data removal for the unknown users
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    activatePersonalDataRemoval: function (controller)
    {
        var userTargets = controller.getMatchingTargets();
        if (userTargets != null && userTargets.length > 0)
        {
            // ask for confirmation before removal
            Ametys.Msg.confirm("{{i18n PLUGINS_ADMIN_UNKNOWN_USERS_ACTION_ACTIVATE_PERSONAL_DATA_REMOVAL_LABEL}}",
                "{{i18n PLUGINS_ADMIN_UNKNOWN_USERS_ACTION_ACTIVATE_PERSONAL_DATA_REMOVAL_CONFIRM}}",
                Ext.bind(this._doActivatePersonalDataRemoval, this, [userTargets, controller], 1),
                this
            );
        }
    },

    /**
     * @private
     */
    _doActivatePersonalDataRemoval: function(buttonId, targets, controller)
    {
        if (buttonId == 'yes' && targets.length > 0)
        {
            var users = Ext.Array.map(targets, target => ({
                    login: target.getParameters().login,
                    populationId: target.getParameters().populationId
                })
            );
            
            controller.serverCall("activatePersonalDataRemoval", [users], this._setPersonalDataRemovalCb, {
                arguments: true,
                refreshing: true,
                errorMessage: true
            });
        }
    },

    /**
     * Deactivate the personal data removal for the users
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    deactivatePersonalDataRemoval: function (controller)
    {
        var userTargets = controller.getMatchingTargets();
        if (userTargets != null && userTargets.length > 0)
        {
            var users = Ext.Array.map(userTargets, target => ({
                    login: target.getParameters().login,
                    populationId: target.getParameters().populationId
                })
            );
            
            controller.serverCall("deactivatePersonalDataRemoval", [users], this._setPersonalDataRemovalCb, {
                arguments: false,
                refreshing: true,
                errorMessage: true
            });
        }
    },

    _setPersonalDataRemovalCb: function(users, forget)
    {
        Ametys.notify({
            title: "{{i18n PLUGINS_ADMIN_UNKNOWN_USERS_ACTION_PERSONAL_DATA_REMOVAL_NOTIFICATION_TITLE}}",
            description: forget
                ? users.length == 1
                    ? "{{i18n PLUGINS_ADMIN_UNKNOWN_USERS_ACTION_ACTIVATE_PERSONAL_DATA_REMOVAL_SUCCESS}}"
                    : "{{i18n PLUGINS_ADMIN_UNKNOWN_USERS_ACTION_ACTIVATE_PERSONAL_DATA_REMOVAL_SUCCESS_MULTIPLE}}.".replace("{0}", users.length)
                :users.length == 1
                    ? "{{i18n PLUGINS_ADMIN_UNKNOWN_USERS_ACTION_DEACTIVATE_PERSONAL_DATA_REMOVAL_SUCCESS}}"
                    : "{{i18n PLUGINS_ADMIN_UNKNOWN_USERS_ACTION_DEACTIVATE_PERSONAL_DATA_REMOVAL_SUCCESS_MULTIPLE}}.".replace("{0}", users.length),
            iconGlyph: forget ? 'ametysicon-multimedia-command-play' : 'ametysicon-datetime-clock'
        });
        
        var targets = Ext.Array.map(users, user => ({
            id: Ametys.message.MessageTarget.UNKNOWN_USER,
            parameters: {
                login: user.login,
                populationId: user.populationId,
                forget: forget
            }
        }))
        
        Ext.create('Ametys.message.Message', {
            type: Ametys.message.Message.MODIFIED,
            targets: targets
        });
    }
});