/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/*
 * Compute a version component for each sql table used by a group directory.
 * If two directories use the same table (highly improbable) only one version
 * will be computed. This allow the use of script that will fail on re-execution
 * 
 * By creating a version for each table, we unsure to migrate table that belongs
 * to groups that were unavailable during a previous migration (because
 * we were in safe mode for exemple).
 */

const results = new java.util.HashMap();
const alreadyAddedTables = []; // In case multiple directories use the same table

const sqlDataSourceManager = Ametys.serviceManager.lookup(org.ametys.core.datasource.SQLDataSourceManager.ROLE);

const versionStorageEP = Ametys.serviceManager.lookup(org.ametys.core.migration.version.storage.VersionStorageExtensionPoint.ROLE);
const versionStorage = versionStorageEP.getExtension('sql');

const groupDirectories = Ametys.serviceManager.lookup('org.ametys.core.group.GroupDirectoryDAO').getGroupDirectories();
for (let directory of groupDirectories)
{
    if (directory.getGroupDirectoryModelId() == "org.ametys.plugins.core.group.directory.Jdbc")
    {
        const datasourceId = directory.getParameterValues().get("runtime.groups.jdbc.datasource");
        
        // resolve the potential indirection due to the default datasource
        const actualDatasourceId = sqlDataSourceManager.getDataSourceDefinition(datasourceId).getId();
        
        const tableName = directory.getParameterValues().get("runtime.groups.jdbc.list.table");
        if (!alreadyAddedTables.includes(actualDatasourceId + "#" + tableName))
        {
            alreadyAddedTables.push(actualDatasourceId + "#" + tableName);
            
            const connection = org.ametys.core.datasource.ConnectionHelper.getConnection(actualDatasourceId);
            if (org.ametys.core.script.SQLScriptHelper.tableExists(connection, tableName))
            {
                const tableNameComposition = directory.getParameterValues().get("runtime.groups.jdbc.composition.table");
                const versionComponentId = `${componentId}_${tableName}`;
                const configuration = new org.ametys.core.migration.version.storage.SqlVersionStorage.SqlVersionConfiguration(actualDatasourceId);
                
                const versions = versionStorage.getVersions(versionComponentId, configuration, component);
                for (version of versions)
                {
                    version.addAdditionalValue("TABLENAME", tableName);
                    version.addAdditionalValue("TABLENAME_COMPOSITION", tableNameComposition);
                }

                const dsDef = sqlDataSourceManager.getDataSourceDefinition(actualDatasourceId);
                const dsKey = dsDef.getName();
              
                if (results.get(dsKey) === null)
                {
                    results.put(dsKey, {});
                }
                    
                results.get(dsKey)[tableName] = {
                    "id": `${componentId}_${actualDatasourceId}_${tableName}`,
                    "versions": versions,
                    "componentId": versionComponentId,
                    "versionConfiguration": configuration,
                    "additionalValues": {"TABLENAME": tableName, "TABLENAME_COMPOSITION": tableNameComposition}
                };
            }
            
            org.ametys.core.datasource.ConnectionHelper.cleanup(connection);
        }
    }
}

return results;
