/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This tool is the abstract tool for exploring folder/files on system hard drive.
 * 
 * Creates your own Tool class by inheriting this one and define at least the following methods: 
 * 
 *  * #createTree, 
 * 
 */
Ext.define('Ametys.file.AbstractFileExplorerTool', {
    extend: "Ametys.tool.Tool",
    
    /**
     * @property {Ext.tree.Panel} _tree The tree panel displaying the parameters files
     * @private
     */
    
    createPanel: function()
    {
        this._tree = this.createTree();
        
        this._tree.on('itemdblclick', Ext.bind(this._openDblClick, this));
        this._tree.on('selectionchange', Ext.bind(this.sendCurrentSelection, this));
        
        return this._tree;
    },
    
    /**
     * @protected
     * @template
     * Create the explorer files tree
     * @return {Ext.tree.Panel} the tree
     */
    createTree: function ()
    {
        throw new Error("The method #createTree is not implemented in " + this.self.getName());
    },
    
    getMBSelectionInteraction: function() 
    {
        return Ametys.tool.Tool.MB_TYPE_ACTIVE;
    },
    
    sendCurrentSelection: function()
    {
        var selection = this._tree.getSelectionModel().getSelection();
        var node = selection.length > 0 ? selection[0] : null;
        
        var target = null;
        if (node != null)
        {
            target = this._tree.getMessageTargetConfiguration(node, true);
        }
        
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.SELECTION_CHANGED,
            
            targets: target
        });
    },
    
    setParams: function(params)
    {
        this.callParent(arguments);
        this.refresh();
    },

    refresh: function()
    {
        this.showRefreshing();
        
        this._tree.getStore().load({
            node: this._tree.getRootNode(),
            callback: this._refreshCb,
            scope: this
        });
    },
    
    /**
     * Callback function called after #refresh is processed. Selects the first node in the tree.
     * @private
     */
    _refreshCb: function()
    {
        // Select first node
        this._tree.getSelectionModel().select(this._tree.getRootNode().firstChild);
        this.showRefreshed();
    },

    /**
     * @private
     * This listener is called before the dblclick event on an node is processed. 
     * Call #openFile is node is a file resource 
     * @param {Ext.tree.Panel} tree The tree
     * @param {Ext.data.Model} record The record that belongs to the item
     */
    _openDblClick: function(tree, record)
    {       
        if (record.get('type') == Ametys.file.AbstractFileExplorerTree.TYPE_RESOURCE)
        {
            this.openFile(record);
        }
    },
    
    /**
     * @protected
     * Open a file resource
     * @param {Ext.data.Model} record The file record to open
     */
    openFile: function(record)
    {
        return;
    },
        
    /**
     * Reload a node
     * @param {String} path The path of the node to reload
     * @param {Function} callback The callback function. Can be null.
     * @param {Array} args The arguments to the callback function
     * @private
     */
    refreshNode: function(path, callback, args)
    {
        this._tree.refreshNode(path, callback, args);
    }
});

