/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class controls a ribbon button representing the pending changes state of a system file being editing.
 * @private
 */
Ext.define(
    "Ametys.file.FileEditorController",
    {
        extend: "Ametys.ribbon.element.ui.ButtonController",

        /**
         * @cfg {String} editor-tool-prefix The tool prefix of editor tool
         */
        
        constructor: function(config)
        {
            this.callParent(arguments);

            this._editorToolPrefix = this.getInitialConfig("editor-tool-prefix");
            
            Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onFileModified, this);
            Ametys.message.MessageBus.on(Ametys.message.Message.MODIFYING, this._onFileModifying, this);
        },

        /**
         * Called when the state of the button is changed.
         * Check if the skin editor tool is dirty, and stay disabled if not.
         */
        updateState: function()
        {
            var target = this.getMatchingTargets()[0];
            if (target)
            {
                var path = target.getParameters().path.replace(/\//g, "-");
                var fileTool = Ametys.tool.ToolsManager.getTool(this._editorToolPrefix + "$" + path);
                if (fileTool)
                {
                    this.setDisabled(!fileTool.isDirty());
                }
            }
        },

        /**
         * Listener when the selection is being modified.
         * Enable the controller.
         * @param {Ametys.message.Message} message The selection changed message.
         * @protected
         */
        _onFileModifying: function(message)
        {
            message = message || Ametys.message.MessageBus.getCurrentSelectionMessage();

            var target = message.getTarget(this._selectionTargetType);
            if (target)
            {
                this.enable();
            }
        },

        /**
         * Listener when the selection has been modified.
         * Disable the controller.
         * @param {Ametys.message.Message} message The selection changed message.
         * @protected
         */
        _onFileModified: function(message)
        {
            message = message || Ametys.message.MessageBus.getCurrentSelectionMessage();

            var target = message.getTarget(this._selectionTargetType);
            if (target && message.getParameters().major)
            {
                this.disable();
            }
        }

    });
