/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This UI helper provides a dialog box to display errors while saving
 * See #showErrorDialog method
 * @private
 */
Ext.define('Ametys.form.SaveHelper.SaveErrorDialog', {
	singleton: true,
	
	/**
	 * @private
	 * @property {Boolean} _initialized Is the dialog already initialized?
	 */
	_initialized: false,
	
	/**
     * @private
     * @property {Function} _warningsCallback the callback function invoked upon the user's choice on warnings treatment. See #showWarningDialog
     */
	
	/**
     * @private
     * @property {Object} _nextErrors If the #showErrorDialog or #showWarningDialog is called while a dialog is already opened, store the message and display it when the dialog is closed
     */
	_nextErrors : [],
		
	/**
	 * Show the error dialog box
	 * @param {String} title The dialog box's title. Can be null to have a default 'save' title
	 * @param {String} msg The dialog box's main message
	 * @param {String} detailedMsg The detailed message (scrolling if too large)
	 */
	showErrorDialog: function (title, msg, detailedMsg)
	{
		this._delayedInitialized();
		
		if (this._errorDialog.isVisible() || this._warningDialog.isVisible())
		{
            this._nextErrors.push({
                type: 'error',
                title: title,
                msg: msg,
                detailedMsg: detailedMsg 
            });
        }
        else
        {
    		this._errorDialog.setTitle (title || "{{i18n PLUGINS_CORE_UI_SAVE_ACTION_FAILED_TITLE}}");
    		
    		this._errorDialog.items.get(0).update(msg);
    		this._errorDialog.items.get(1).update(detailedMsg);
    		
    		this._errorDialog.show();
        }
	},
	
	/**
     * Show the warning dialog box
     * @param {String} title The dialog box's title. Can be null to have a default 'save' title
     * @param {String} msg The dialog box's main message
     * @param {String} detailedMsg The detailed message (scrolling if too large)
     * @param {String} question The the question message (save with the warnings or not)
     * @param {Function} callback the callback function
     */
    showWarningDialog: function (title, msg, detailedMsg, question, callback)
    {
        this._delayedInitialized();
        
        if (this._errorDialog.isVisible() || this._warningDialog.isVisible())
        {
            this._nextErrors.push({
                type: 'warning',
                title: title,
                msg: msg,
                detailedMsg: detailedMsg,
                question: question,
                callback: callback
            });
        }
        else
        {
            this._warningDialog.setTitle (title || "{{i18n PLUGINS_CORE_UI_SAVE_ACTION_WARNING_TITLE}}");
                
            this._warningDialog.items.get(0).update(msg);
            this._warningDialog.items.get(1).update(detailedMsg);
            this._warningDialog.items.get(2).update(question);
            
            this._warningsCallback = callback;

            this._warningDialog.show();
        }
    },
	
	/**
     * @private
     * When closing the dialog, should be displayed another one?
     */
	_onHide: function()
	{
        if (this._nextErrors.length > 0)
        {
            let me = this;
            let err = this._nextErrors.shift();
            if (err.type == 'error')
            {
                window.setTimeout(function() { me.showErrorDialog(err.title, err.msg, err.detailedMsg); }, 1);
            }
            else
            {
                window.setTimeout(function() { me.showWarningDialog(err.title, err.msg, err.detailedMsg, err.question, err.callback); }, 1);
            }
        }
    },
	
	/**
	 * @private
	 * Initialize the error dialog box
	 */
	_delayedInitialized: function ()
	{
		if (!this._initialized)
		{
            let dialogsCommonCfg = {
                width: 450,
                maxHeight: 280,
                layout: { 
                    type: 'vbox',
                    align: 'stretch'
                },
                
                defaultFocus: 0,
                
                closeAction: 'hide'
            }
            
            let errorDialogCfg = Ext.merge (dialogsCommonCfg, {
                title: "{{i18n PLUGINS_CORE_UI_SAVE_ACTION_FAILED_TITLE}}",
                iconCls: 'ametysicon-letter-x5',
                
                items: [{
                        xtype: 'container',
                        cls: 'save-error-dialog-text',
                        html: ''
                    },
                    {
                        xtype: 'container',
                        cls: 'save-error-dialog-details',
                        flex: 1,
                        scrollable: true,
                        html: ''
                    }
                ],
                
                listeners: {
                    scope: this,
                    'hide': this._onHide
                },
                
                buttons: [{
                    text : "{{i18n PLUGINS_CORE_UI_SAVE_ACTION_FAILED_OK_BTN}}",
                    handler : function () { this._errorDialog.close()},
                    scope: this
                }]
            });
            
			this._errorDialog = Ext.create('Ametys.window.DialogBox', errorDialogCfg);
			
			let warningDialogCfg = Ext.merge (dialogsCommonCfg, {
                title: "{{i18n PLUGINS_CORE_UI_SAVE_ACTION_WARNING_TITLE}}",
                iconCls: 'ametysicon-sign-caution',
                
                items: [{
                        xtype: 'container',
                        cls: 'save-warning-dialog-text',
                        html: ''
                    },
                    {
                        xtype: 'container',
                        cls: 'save-warning-dialog-details',
                        flex: 1,
                        scrollable: true,
                        html: ''
                    },
                    {
                        xtype: 'container',
                        cls: 'save-warning-dialog-question',
                        html: ''
                    }
                ],
                
                listeners: {
                    scope: this,
                    close: function() {
                        if (Ext.isFunction(this._warningsCallback))
                        {
                            this._warningsCallback(false);
                            this._warningsCallback = null;
                        }
                    },
                    'hide': this._onHide
                },
                
                buttons: [{
                    text: "{{i18n PLUGINS_CORE_UI_SAVE_ACTION_WARNING_YES_BTN}}",
                    handler: function () 
                    { 
                        this._warningsCallback(true);
                        this._warningsCallback = null;
                        this._warningDialog.close();
                    },
                    scope: this
                }, {
                    text: "{{i18n PLUGINS_CORE_UI_SAVE_ACTION_WARNING_NO_BTN}}",
                    handler: function () 
                    { 
                        this._warningsCallback(false);
                        this._warningsCallback = null;
                        this._warningDialog.close();
                    },
                    scope: this
                }]
            });
            
            this._warningDialog = Ext.create('Ametys.window.DialogBox', warningDialogCfg);

			this._initialized = true;
		}
	}
});