/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Provides a widget for datetime input field.<br>
 * This widget is available for fields of type Ametys.form.WidgetManager#TYPE_DATE.<br> 
 * It does NOT handle multiple values.<br>
 * 
 * The format used for date is ISO 8601
 */
Ext.define('Ametys.form.widget.DateTime', {
    extend: "Ametys.form.field.DateTime",
    
    canDisplayComparisons: true,

    /**
     * @cfg {Boolean} [initWithCurrentDate=false] True to initialize the field with the current date.
     */
        
    constructor: function (config)
    {
        let maxHeight = Ametys.form.WidgetManager.getWidgetMaxHeight(config);
        
        config.dateConfig = Ext.apply(config.dateConfig || {}, {
            value: this._getDateValue(config),
            format: Ext.Date.patterns.LongDate,
            altFormats: 'c',
            submitFormat: Ext.Date.patterns.ISO8601Date,
            maxHeight: maxHeight
        });
        
        config.timeConfig = Ext.apply(config.timeConfig || {}, {
            value: this._getDateValue(config),
            format: "H:i"
        });
        
        delete config.value;
        
        this.callParent(arguments);
    },
    
    getReadableValue: function ()
    {
        return this.getValue() ? Ext.Date.format(this.getValue(), Ext.Date.patterns.FriendlyDateTime) : "";
    },

    /**
     * @private
     * Function invoked when the combo box is clicked
     */
    _getDateValue: function(config)
    {
        var initWithCurrentDate = Ext.isBoolean(config.initWithCurrentDate) ? config.initWithCurrentDate : config.initWithCurrentDate === "true";
        var value = config.value;
        if (!config.value && initWithCurrentDate)
        {
            value = new Date();
        }
        else if (!Ext.isDate(config.value))
        {
            value = Ext.Date.parse(config.value, this.submitFormat);
        }
        return value;
    },
    
       
    /**
      * When used in readonly mode, settting the comparison value will display ins/del tags
      * @param {String} otherValue The value to compare the current value with
      * @param {boolean} base When true, the value to compare is a base version (old) ; when false it is a future value
      */
    setComparisonValue: function(otherValue, base)
    {
         if (base)
         {
             this._baseValue = otherValue || null;
             this._futureValue = undefined;
         }
         else
         {
             this._baseValue = undefined;
             this._futureValue = otherValue || null;
         }
         this.setValue(this.getValue());
    },
    
    setValue: function(value)
    {
        this.callParent(arguments);

        var v = this.getValue();
        if (this._baseValue !== undefined)
        {
            this.toggleCls("ametys-field-datetime-date-newvalue", v != null && this._dateToString(v) != this._dateToString(this._baseValue));
            this.toggleCls("ametys-field-datetime-time-newvalue", v != null && this._timeToString(v) != this._timeToString(this._baseValue));
        }
        else if (this._futureValue !== undefined)
        {
            this.toggleCls("ametys-field-datetime-date-oldvalue", v != null && this._dateToString(v) != this._dateToString(this._futureValue));
            this.toggleCls("ametys-field-datetime-time-oldvalue", v != null && this._timeToString(v) != this._timeToString(this._futureValue));
        }
    },
    
    _dateToString: function(d)
    {
        if (!d)
        {
            return "";
        }
        
        if (Ext.isString(d)) 
        {
            d = Ext.Date.parse(d);
        }
        
        return Ext.Date.format(d, Ext.Date.patterns.ISO8601Date);
    },
    
    _timeToString: function(d)
    {
        if (!d)
        {
            return "";
        }
        
        if (Ext.isString(d)) 
        {
            d = Ext.Date.parse(d);
        }
        
        return Ext.Date.format(d, Ext.Date.patterns.ISO8601DateTime).substring(11);
    }
});
