/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/**
 * Provides a widget for text field<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_STRING.<br>
 * It does handle multiple values (see {@link #cfg-multiple}) using separated by commas text. 
 */
Ext.define('Ametys.form.widget.TextComparison', {
    extend: "Ext.form.field.Display",
    
    canDisplayComparisons: true,
    focusable: true,
    
    constructor: function(config)
    {
        config.cls = 'ametys-field-text-comparison' + (config.cls ? ' ' + config.cls : '');
        this.callParent(arguments);
    },
    
    afterRender: function(e)
    {
        this.bodyEl.addCls('x-form-trigger-wrap-default');
        this.inputEl.addCls('x-form-text-default');
        return this.callParent(arguments);
    },
    
    onFocus: function(e)
    {
        this.bodyEl.addCls('x-form-trigger-wrap-focus')
        return this.callParent(arguments);
    },

    onBlur: function(e)
    {
        this.bodyEl.removeCls('x-form-trigger-wrap-focus')
        return this.callParent(arguments);
    },
    
    /**
     * @private
     */
    _escapeValue: function(v)
    {
        if (v !== null 
            && v !== undefined)
        {
            // convert v to string if necessary (can be a number by inheritance)
            return (v = "" + v).replaceAll('&', '&amp;').replaceAll('<', '&lt;')
        }
        else 
        {
            return "";
        }
    },
        
    /**
     * @private
     */
    _encodeValue: function(v)
    {
        if (Array.isArray(v))
        {
            var encodedValue = [];
            for (var i = 0; i < v.length; i++)
            {
                encodedValue.push(this._escapeValue(v[i]));
            }
            return encodedValue.join(", ");
        }
        else
        {
            return this._escapeValue(v);
        }
    },
    
    setValue: function(value)
    {
        this._rawValue = value;
        
        var toDisplayValue = this._encodeValue(value);
        
        if (this._baseValue !== undefined)
        {
            var newValue = HtmlDiff.execute(this._encodeValue(this._baseValue), toDisplayValue);
            toDisplayValue = newValue.replace(/<del [^>]*>.*?<\/del>/g, "");
        }
        else if (this._futureValue !== undefined)
        {
            var newValue = HtmlDiff.execute(toDisplayValue, this._encodeValue(this._futureValue));
            toDisplayValue = newValue.replace(/<ins [^>]*>.*?<\/ins>/g, "");            
        }
        
        this.callParent([toDisplayValue]);
    },
    
    /**
      * When used in readonly mode, settting the comparison value will display ins/del tags
      * @param {String} otherValue The value to compare the current value with
      * @param {boolean} base When true, the value to compare is a base version (old) ; when false it is a future value
      */
    setComparisonValue: function(otherValue, base)
    {
         if (base)
         {
             this._baseValue = otherValue || null;
             this._futureValue = undefined;
         }
         else
         {
             this._baseValue = undefined;
             this._futureValue = otherValue || null;
         }
         this.setValue(this._rawValue);
    },
});