/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class for the actions related to the parameter files tool.
 * @private
 */
Ext.define('Ametys.plugins.coreui.parameter.files.ParameterActions', {
    singleton: true,

    /**
     * Restart Ametys application to take into account the parameter files modifications
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    restart: function(controller) 
    {       
        Ametys.Msg.confirm("{{i18n PLUGINS_CORE_UI_PARAMETERS_RESTART_LABEL}}", 
                "{{i18n PLUGINS_CORE_UI_PARAMETERS_RESTART_CONFIRM}}", 
                Ext.bind(this._restartConfirm, this, [controller], 1),
                this
        );
    },

    /**
     * Callback function invoked after the 'restart' confirm box is closed
     * @param {String} answer Id of the button that was clicked
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the #restart function
     * @private
     */
    _restartConfirm: function(answer, controller)
    {
        if (answer == 'yes')
        {       
            var url = Ametys.getPluginDirectPrefix('admin') + "/restart",
                result = null,
                ex = '';
            
            try
            {
                result = Ext.Ajax.request({url: url, async: false});   
            }
            catch (e)
            {
                ex = e;
            }
            
            Ext.getBody().unmask();
            
            if (result == null)
            {
                Ametys.log.ErrorDialog.display({
                    title: "{{i18n PLUGINS_CORE_UI_PARAMETERS_RESTART_FAILED}}", 
                    text: "{{i18n PLUGINS_CORE_UI_PARAMETERS_RESTART_FAILED_TEXT}}",
                    details: ex,
                    category: "Ametys.plugins.coreui.parameter.files"
                });
                return;
            }
            
            // Reload application
            document.location.reload(true);
        }
    },
   
   /**
     * Action function to be called by the controller.
     * Will save the modifications
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    save: function(controller)
    {
        this._startSave(controller, false);
    },
    
    /**
     * Action function to be called by the controller.
     * Will save the modifications and quit the editor tool
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    saveAndQuit: function(controller)
    {
        this._startSave(controller, true);
    },
   
   /**
    * Save the content of the current editor tool
    * Sends a {@link Ametys.message.Message#MODIFIED} event on success.
    * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
    * @param {Boolean} quit true to quit editor tool after saving modifications
    */
   _startSave: function(controller, quit)
   {
       var editorTool = Ametys.tool.ToolsManager.getFocusedTool();
       if (editorTool != null)
       {
           this._doSave(editorTool, editorTool.getFileName(), editorTool.getFilePath(), quit, controller);
       }
   },
   
   /**
    * Save the current modifications
    * @param {Ametys.plugins.coreui.parameter.files.tool.ParameterFileEditorTool} editorTool The file editor tool
    * @param {String} fileName The name of the file
    * @param {String} filePath The path of the file
    * @param {Boolean} quit true to quit editor tool after saving modifications
    * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
    * @private
    */
   _doSave: function(editorTool, fileName, filePath, quit, controller)
   {       
       controller.serverCall(
           'saveParameters', [filePath, editorTool.getText()] ,Ext.bind (this._saveCb, this), {arguments : {tool: editorTool, name: fileName, path: filePath, quit: quit} },
           {
                waitMessage: {
                    target: editorTool.getContentPanel(),
                    msg: "{{i18n PLUGINS_CORE_UI_PARAMETERS_EDITOR_SAVING_MSG}}"
                }, 
                errorMessage: {
                    msg: "{{i18n PLUGINS_CORE_UI_PARAMETERS_EDITOR_SAVE_ERROR}}",
                    category: "Ametys.plugins.coreui.parameter.files"
                }
           }
       );
   },
   
   /**
     * @private
     * Callback function called after save action is processed
     * @param {Object[]} response The server response. response = null is everything is ok
     * @param {String} response.msg the message to show if there is one to show
     * @param {Object[]} args The callback parameters.
     * @param {Ametys.plugins.coreui.parameter.files.tool.ParameterFileEditorTool} args.editorTool The file editor tool
     * @param {String} args.name the name of the file
     * @param {String} args.path filePath the path of the file
     * @param {Boolean} args.quit true if we quit the editor tool after saving the modifications
     */
   _saveCb: function(response, args)
   {    
       if (!response.success &&  response.error == 'unknown-file')
       {
           Ametys.Msg.show({
               title: "{{i18n PLUGINS_CORE_UI_PARAMETERS_EDITOR_SAVE_LABEL}}",
               msg: "{{i18n PLUGINS_CORE_UI_PARAMETERS_EDITOR_SAVE_UNKNOWN_FILE}}",
               buttons: Ext.Msg.OK,
               icon: Ext.MessageBox.ERROR
           });
           return;
       }
       
       if (response.isI18n)
       {
           Ametys.Msg.show({
               title: "{{i18n PLUGINS_CORE_UI_PARAMETERS_EDITOR_SAVE_LABEL}}",
               msg: "{{i18n PLUGINS_CORE_UI_PARAMETERS_EDITOR_SAVE_NEED_WAIT}}",
               buttons: Ext.Msg.OK,
               icon: Ext.MessageBox.INFO
           });
       }
       
       Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            parameters: {major: true},
            targets: {
              id: Ametys.message.MessageTarget.PARAM_FILE,
              parameters: {
                  'name': args.name, 
                  'path': args.path
              }
            }
        });
       
       if (args.quit)
       {
           args.tool.close();
       }
   },
   
   /**
     * Action function to be called by the controller.
     * Quit the edition without saving
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    unsave: function(controller)
    {
        Ametys.Msg.confirm("{{i18n PLUGINS_CORE_UI_PARAMETERS_EDITOR_UNSAVE_LABEL}}", 
                "{{i18n PLUGINS_CORE_UI_PARAMETERS_EDITOR_UNSAVE_CONFIRM}}", 
                Ext.bind(this._unsaveConfirm, this, [controller], 1),
                this
        );
    },
    
    /**
     * Callback function invoked after the 'unsave' confirm box is closed
     * @param {String} answer Id of the button that was clicked
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the #unsave function
     * @private
     */
    _unsaveConfirm: function(answer, controller)
    {
        if (answer == 'yes')
        {
            var editorTool = Ametys.tool.ToolsManager.getFocusedTool();
            if (editorTool != null)
            {
                editorTool.close();
            }
        }
    },
    
    // ------------------------ FOLDERS -------------------------------//
    
    /**
     * Function called to add a new folder
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    addFolder: function (controller)
    {
        var targets = controller.getMatchingTargets();
        
        if (targets.length > 0)
        {
            controller.serverCall(
                    'addFolder', [targets[0].getParameters().path, "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_ADD_FOLDER_NEW}}"], Ext.bind(this._addFolderCb,this),
                    {
                        errorMessage: { msg: "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_ADD_FOLDER_ERROR}}", category: Ext.getClassName(this) + '.addFolder'},
                        refreshing: true
                    }
            );
        }
    },
    
    /**
     * Callback invoked when adding a folder
     * @param {Object[]} response The JSON response from the server
     * @param {String} response.path the path of the folder
     * @param {String} response.parentPath the path of the parent 
     * @param {String} response.name the name of the folder
     */
    _addFolderCb : function(response) 
    {
        if (response.success)
        {
            var path = response['path'];
            var name = response['name'];
            
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.CREATED,
                targets: {
                    id: Ametys.message.MessageTarget.PARAM_FOLDER,
                    parameters: { 
                        path: path,
                        name: name
                    }
                }
            });
        }
    },
    
    /**
     * Function called to delete a folder
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    removeFolder: function(controller)
    {
        var targets = controller.getMatchingTargets();
        
        if (targets.length > 0)
        {
            Ametys.Msg.confirm("{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_DELETE_FOLDER_TITLE}}", 
                    "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_DELETE_FOLDER_CONFIRM}}", 
                    function(btn) {
                        this._doRemove(btn, targets[0], controller);
                    }, 
                    this);
        }
    },
    
    /**
     * @private
     * Internal callback for the #removeFolder function
     * @param {String} answer The id of the button pressed.
     * @param {Ametys.message.MessageTarget} target The target
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function     
     */
    _doRemove: function(answer, target, controller)
    {
        if (answer == 'yes')
        {
            var errorMsg = target.getId() == Ametys.message.MessageTarget.PARAM_FOLDER ? "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_DELETE_FOLDER_ERROR}}" : "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_DELETE_FILE_ERROR}}";
            
            controller.serverCall(
                    'deleteFile', [target.getParameters().path], Ext.bind(this._doRemoveCb,this), {arguments: {target: target}},
                    {
                        errorMessage: { msg: errorMsg, category: Ext.getClassName(this) + '.removeFolder'},
                        refreshing: true
                    }
            );
        }
    },
    
    /**
     * Callback function after deletion.
     * @param {Object[]} result the server's response
     * @param {Object} params The callback arguments
     */
    _doRemoveCb: function(result, params)
    {
        if (result.success)
        {
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.DELETED,
                targets: params.target
            });
        }
    },
    
    /**
     * Function called to rename a folder
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    renameFolder: function(controller)
    {
        var targets = controller.getMatchingTargets();
        
        if (targets.length > 0)
        {
            var name = targets[0].getParameters().name;
            
            Ametys.Msg.prompt(
                "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FOLDER_TITLE}}", 
                "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FOLDER_MSG}}", 
                function(btn, text) {
                    if (btn == 'ok' && text != name) 
                    {
                        this.rename(targets[0].getParameters().path, text, 'collection');
                    }
                },
                this, 
                false,
                name
            );
        }
    },
    
    // ----------------------------- FILES -------------------------------//
    /**
     * Determines if a file can be edited online by its name
     * @param {String} name The file name
     */
    isFileEditableOnline: function(name)
    {
        var extension = Ametys.file.AbstractFileExplorerTree.getFileExtension(name).toLowerCase();
        return extension == 'js' || extension == 'html' || extension == 'xhtml' || extension == 'css' || extension == 'xml' || extension == 'xsl' || extension == 'xslt' || extension == 'txt';
    },
    
    /**
     * Open or donwload a file
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    openFile: function(controller)
    {
        var targets = controller.getMatchingTargets();
        if (targets.length > 0)
        {
            var target = targets[0];
            this.open (target.getParameters().name, target.getParameters().path);
        }
    },
    
    /**
     * Open or download a file
     * @param {String} name The file name
     * @param {String} path The path of the file
     */
    open: function(name, path)
    {
        if (this.isFileEditableOnline(name))
        {
            var params = {};
            params.id = path.replace(/\//g, "-");
            params.filename = name
            params.path = path;
            
            var fileTool = Ametys.tool.ToolsManager.getTool("uitool-file-editor$" + params.id);
            if (fileTool)
            {
                fileTool.focus();
            }
            else
            {
                Ametys.tool.ToolsManager.openTool("uitool-file-editor", params);
            }
            
        }
        else
        {
            // Download uneditable file
            var url = Ametys.getPluginDirectPrefix('core-ui') + '/param/download/' + path;
            window.open(url);
        }
    },
    
    /**
     * Function called to add or update a file
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    addFile: function(controller)
    {
        var targets = controller.getMatchingTargets();
        if (targets.length > 0)
        {
            Ametys.plugins.coreui.parameter.files.UploadFile.open(targets[0].getParameters().path, Ext.bind(this._addFileCb, this));
        }
    },
    
    /**
     * @private
     * Callback function called after #addFile action is processed
     * @param {String} name The name of uploaded file
     * @param {String} path The path of uploaded file
     * @param {String} parentPath The parent path of uploaded file
     */
    _addFileCb: function(name, path, parentPath)
    {
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.CREATED,
            targets: {
                id: Ametys.message.MessageTarget.PARAM_FILE,
                parameters: { 
                    path: path,
                    name: name
                }
            }
        });
    },
    
    /**
     * Function called to remove a file
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    removeFile: function(controller)
    {
        var targets = controller.getMatchingTargets();
        
        if (targets.length > 0)
        {
            Ametys.Msg.confirm("{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_DELETE_FILE_TITLE}}", 
                    "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_DELETE_FILE_CONFIRM}}", 
                    function(btn) {
                        this._doRemove(btn, targets[0], controller);
                    }, 
                    this);
        }
    },
    
    /**
     * Function called to remove a file
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    renameFile: function(controller)
    {
        var targets = controller.getMatchingTargets();
        if (targets.length > 0)
        {
            var name = targets[0].getParameters().name;
            var oldPath = targets[0].getParameters().path;
            
            Ametys.Msg.prompt(
                "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_PROMPT_TITLE}}", 
                "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_PROMPT_MSG}}", 
                function(btn, text) {
                    if (btn == 'ok' && text != name) 
                    {
                        this.rename(targets[0].getParameters().path, text, 'resource');
                    }
                },
                this, 
                false,
                name
            );
        }
    },
    
    /**
     * Rename a file or folder
     * @param {String} path The path of file to rename
     * @param {String} name The new name
     * @param {String} type The type of resource
     * @param {Function} [callback] Callback function. The function receives the following parameters:
     * @param {Boolean} callback.success true if the renaming was successful
     * @param {String} callback.path The path of the renamed file
     * @param {String} callback.name The name of the renamed file
     */
    rename: function(path, name, type, callback)
    {
        var name = Ext.String.trim(name);
        if (name == '' || !/^[^\\/:*?"<>|]*$/.test(name))
        {
            var alertMsg = type == 'resource' ? "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FILE_INVALID_CHARACTERS}}" : "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FOLDER_INVALID_CHARACTERS}}"
            Ametys.Msg.alert(
                "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FOLDER_FILE}}",
                alertMsg
            );
            
            if (Ext.isFunction(callback))
            {
                callback(false, path, name);
            }
            return false;
        }
        
        var errorMsg = type == 'resource' ? "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FILE_ERROR}}" : "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FOLDER_ERROR}}";

        Ametys.data.ServerComm.callMethod({
            role: 'org.ametys.core.ui.RibbonControlsManager',
            id: 'org.ametys.plugins.core.ui.parameter.files.Folder.rename',
            methodName: 'renameFile',
            parameters: [path, name],
            callback: {
                handler: this._renameCb,
                scope: this,
                arguments: {
                    name: name, 
                    type: type, 
                    path: path,
                    callback: callback,
                    messageTargetType: type == 'resource' ? Ametys.message.MessageTarget.PARAM_FILE : Ametys.message.MessageTarget.PARAM_FOLDER
                }
            },
            errorMessage: {
                msg: errorMsg,
                category: this.self.getName()
            },
            waitMessage: true
         });
    },
    
    /**
     * @private
     * Callback invoked after renaming
     * @param {Object} result The server result
     * @param {Object} args the callback arguments
     */
    _renameCb: function(result, args)
    {
        if (result.error == 'already-exist')
        {
            var errorMsg = params.type == 'resource' ? "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FILE_ALREADY_EXISTS}}" : "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FOLDER_ALREADY_EXISTS}}";
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_CORE_UI_PARAMETERS_FILE_RENAME_FOLDER_FILE}}",
                msg: errorMsg,
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
            
            if (Ext.isFunction(args.callback))
            {
                params.callback(false, args.path, args.name);
            }
            return false;
        }
        
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            
            targets: {
                id: args.messageTargetType,
                parameters: { 
                    path: result.path,
                    name: result.name
                }
            }
        });
        
        if (Ext.isFunction(args.callback))
        {
            args.callback(true, result.path, result.name);
        }
    }
    
});
