/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * @private
 * Grid panel with a right target as a first column.
 * This class will handle all the necessary configuration for the first column in order to render the target
 * and group them by type.
 * This class is intended to be used with an Ametys.plugins.coreui.profiles.PermissionTargetStore store
 */
Ext.define('Ametys.plugins.coreui.profiles.PermissionTargetGrid', {
    extend: "Ext.grid.Panel",
    
    /**
     * @cfg {Boolean} [lockFirstColumn=true] When 'true', the first column, with the target, will be locked
     */
    constructor: function(config)
    {
        config.scrollable = config.scrollable || true;
        // Grouping by assignment type
        let features = config.features || [];
        features.push({
            ftype: 'grouping',
            enableGroupingMenu: false,
            expandTip: "",
            collapseTip: "",
            groupHeaderTpl: [
                '{name:this.formatTargetType}', 
                {
                    formatTargetType: Ext.bind(function(type) {
                        switch (type) {
                            case Ametys.plugins.coreui.profiles.PermissionTargetStore.TARGET_TYPE_ANONYMOUS:
                                return "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_TARGET_TYPE_ANONYMOUS}}";
                            case Ametys.plugins.coreui.profiles.PermissionTargetStore.TARGET_TYPE_ANYCONNECTEDUSER:
                                return "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_TARGET_TYPE_ANYCONNECTEDUSERS}}";
                            case Ametys.plugins.coreui.profiles.PermissionTargetStore.TARGET_TYPE_USER:
                                return "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_TARGET_TYPE_USERS}}";
                            case Ametys.plugins.coreui.profiles.PermissionTargetStore.TARGET_TYPE_GROUP:
                                return "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_TARGET_TYPE_GROUPS}}";
                            default:
                                // would never go there
                                return "";
                        }
                    }, this)
                }
            ]
        });
        config.features = features;
        
        config.columns = this.getTargetColumn(config).concat(config.columns || [])
        
        this.callParent([config]);
    },
    
    getTargetColumn: function(config)
    {
        return [{
            stateId: 'grid-first-column',
            text: "",
            locked: config.lockFirstColumn != false,
            lockable: false,
            // FIXME we should use the user fullname instead of sortable name
            dataIndex: "sortableLabel",
            width: 300,
            minWidth: 300,
            hideable: false,
            sortable: true,
            renderer: Ext.bind(this._renderTarget, this)
        }];
    },
    
    /**
     * @private
     * Renderer for the first column, depending on the type of assignement
     * @param {Object} value The data value (the user sortable name or the group label)
     * @param {Object} metaData A collection of metadata about the current cell
     * @param {Ametys.plugins.coreui.profiles.ProfileAssignmentsTool.Entry} record The record
     * @return {String} The html representation
     */
    _renderTarget: function(value, metaData, record)
    {
        var type = record.get('targetType');
        switch (type) {
            case Ametys.plugins.coreui.profiles.PermissionTargetStore.TARGET_TYPE_ANONYMOUS:
                var text = "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_TARGET_TYPE_ANONYMOUS}}";
                return '<span class="ametysicon-carnival23"></span> ' + text;
            case Ametys.plugins.coreui.profiles.PermissionTargetStore.TARGET_TYPE_ANYCONNECTEDUSER:
                var text = "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_TARGET_TYPE_ANYCONNECTEDUSERS}}";
                return '<span class="ametysicon-key162"></span> ' + text;
            case Ametys.plugins.coreui.profiles.PermissionTargetStore.TARGET_TYPE_USER:
                return Ametys.grid.GridColumnHelper.renderUser.apply(this, arguments);
            case Ametys.plugins.coreui.profiles.PermissionTargetStore.TARGET_TYPE_GROUP:
                var text = value + ' (' + record.get('groupId') + ', ' + record.get('groupDirectoryLabel') + ')';
                return '<span class="ametysicon-multiple25"></span> ' + text;
            default:
                return value;
        }
    }
});
