/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Actions for adding/deleting records in the grid of the {@link Ametys.plugins.coreui.profiles.ProfileAssignmentsTool}
 * @private
 */
Ext.define('Ametys.plugins.coreui.profiles.ProfileAssignmentsActions', {
    singleton: true,
    
    /**
     * @property {String} CLIPBOARD_PROFILE_ASSIGNMENTS Id of profile assignments to store in clipboard
     */
    CLIPBOARD_PROFILE_ASSIGNMENTS: "profileAssignments",
    
    /**
     * Add users to the assignment grid of the assignment tool
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    addUsers: function(controller)
    {
    	var tool = this._getTool(controller);
        if (tool != null)
        {
	        Ametys.helper.SelectUser.act({
	            callback: Ext.bind(tool.addUsers, tool)
	        });
        }
    },
    
    /**
     * Add groups to the assignment grid of the assignment tool
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    addGroups: function(controller)
    {
    	var tool = this._getTool(controller);
        if (tool != null)
        {
	        Ametys.helper.SelectGroup.act({
	            callback: Ext.bind(tool.addGroups, tool)
	        });
        }
    },
    
    /**
     * Removes the current selected records of the assignment grid of the assignment tool
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    removeAssignments: function(controller)
    {
        var matchingTargets = controller.getMatchingTargets();
        if (matchingTargets.length > 0)
        {
            var tool = this._getTool(controller);
            var assignments = Ext.Array.map(matchingTargets, function(target) {
                return {
                    id: target.getParameters().id,
                    context: target.getParameters().context
                };
            }, this);
            tool.removeAssignments(assignments);
        }
    },
    
    /**
     * Copy the assignments and inheritance status of the current profile context in clipboard. The assignments should be saved first.
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    copyAssignments: function(controller)
    {
        var tool = this._getTool(controller);
        var currentAssignments = tool.getCurrentAssignments();
        var profileCtxTargets = controller.getMatchingTargets();
        
        if (currentAssignments != null && profileCtxTargets.length > 0)
        {
            Ametys.clipboard.Clipboard.setData (this.CLIPBOARD_PROFILE_ASSIGNMENTS, {
                assignments: currentAssignments,
                inheritanceDisallowed: profileCtxTargets[0].getParameters().inheritanceDisallowed
            });
        }
    },
    
    /**
     * Apply the assignments stored in clipboard to the selected contexts
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    pasteAssignments: function(controller)
    {
        var tool = this._getTool(controller);
        
        var profileCtxTargets = controller.getMatchingTargets();
        if (profileCtxTargets.length > 0 && tool != null)
        {
            var clipboardData = Ametys.clipboard.Clipboard.getData();
	        if (clipboardData.length > 0 && Ametys.clipboard.Clipboard.getType() == this.CLIPBOARD_PROFILE_ASSIGNMENTS)
	        {
                var assignmentsToCopy = clipboardData[0];
                // Apply assignments on modifiable contexts
                var contexts = profileCtxTargets.map(function(target) {
                    if (target.getParameters().modifiable)
                    {
	                    return {
                            context: target.getParameters().context,
                            inheritanceAvailable: target.getParameters().inheritanceAvailable
                        }
                    }
                })
                
                if (contexts.length == 0)
                {
                    // no modifiable contexts
                    Ametys.Msg.show({
		                title: "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_PASTE_ASSIGNMENTS_TITLE}}",
		                msg: "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_PASTE_ASSIGNMENTS_NO_MODIFIABLE_CONTEXT}}",
		                buttons: Ext.Msg.OK,
		                icon: Ext.MessageBox.WARNING
		            });
                    return;
                }
                tool.applyAssignments(contexts, assignmentsToCopy);
            }
            else
            {
                Ametys.Msg.show({
                    title: "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_PASTE_ASSIGNMENTS_TITLE}}",
                    msg: "{{i18n PLUGINS_CORE_UI_TOOL_PROFILE_ASSIGNMENTS_PASTE_ASSIGNMENTS_NO_ASSIGNMENT}}",
                    buttons: Ext.Msg.OK,
                    icon: Ext.MessageBox.WARNING
                });
            }
        }
        
    },
    
    /**
     * Saves the changes made in the assignment tool
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    save: function(controller)
    {
        var tool = this._getTool(controller);
        if (tool != null)
        {
        	tool.saveChanges();
        }
    },
    
    /**
     * Cancel the changes made in the assignment tool
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    cancel: function(controller)
    {
    	var tool = this._getTool(controller);
        if (tool != null)
        {
        	tool.cancelChanges();
        }
    },
    
    /**
     * @private
     * Gets the assignment tool
     * @return {Ametys.tool.Tool} the assignment tool
     */
    _getTool: function(controller)
    {
    	 var toolTarget = controller.getMatchingToolTargets()[0];
         if (toolTarget != null)
         {
        	 return toolTarget.getParameters().tool;
         }
    },
    
    /**
     * Disallow inheritance for the selected context
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    disallowInheritance: function(controller)
    {
        var matchingTargets = controller.getMatchingTargets();
        if (matchingTargets.length > 0)
        {
            var tool = this._getTool(controller);
            if (tool != null)
            {
                tool.disallowInheritance(controller.isPressed());
            }
            
        }
    }
});
