/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Helper to create a grid with profiles as column
 * @private
 */
Ext.define('Ametys.plugins.coreui.profiles.ProfilesGridHelper', {
    statics: {
        /**
         * @readonly
         * @property {String} READER_PROFILE_ID The id of the special profile (READER profile)
         */
        READER_PROFILE_ID: 'READER',
        
        /**
         * Get the column definition based on a list of profiles
         * @param {Object[]} a list of object representing the profiles
         * @param {Function} a renderer to use for the profile columns
         * @param {Object} the scope for the renderer
         * @returns {Object[]} the columns definition
         */
        getProfilesColumns: function (profiles, renderer, scope)
        {
            var columns = [];
            Ext.Array.forEach(profiles, function(profile) {
                columns.push({
                    stateId: 'grid-profile-' + profile.id,
                    text: Ext.String.escapeHtml(profile.label),
                    locked: profile.id == Ametys.plugins.coreui.profiles.ProfilesGridHelper.READER_PROFILE_ID ? true : null,
                    lockable: false,
                    tooltip: Ext.String.htmlEncode(Ext.String.htmlEncode(profile.label)), // Mystery... why do we need to escape the label twice?
                    dataIndex: profile.id,
                    hideable: profile.id != Ametys.plugins.coreui.profiles.ProfilesGridHelper.READER_PROFILE_ID,
                    sortable: false,
                    align: 'center',
                    width: 115,
                    cls: profile.id == Ametys.plugins.coreui.profiles.ProfilesGridHelper.READER_PROFILE_ID ? 'a-column-header-reader' : '',
                    tdCls: profile.id == Ametys.plugins.coreui.profiles.ProfilesGridHelper.READER_PROFILE_ID ? 'a-grid-cell-reader' : '',
                    renderer: renderer,
                    scope: scope,
                    rights: profile.rights
                });
            }, this);
            
            // Reader first then all the profiles alphabetically sorted
            return columns.sort(function(a, b) {
                if (a.dataIndex == Ametys.plugins.coreui.profiles.ProfilesGridHelper.READER_PROFILE_ID)
                {
                    return -1;
                }
                else if (b.dataIndex == Ametys.plugins.coreui.profiles.ProfilesGridHelper.READER_PROFILE_ID)
                {
                    return 1;
                }
                else
                {
                    return Ext.data.SortTypes.asNonAccentedUCString(a.text) < Ext.data.SortTypes.asNonAccentedUCString(b.text) ? -1 : 1;
                }
            });
        },
        
        getColumnsFromPermissions: function(permissions, renderer, scope)
        {
            let profileColumns = [];
            let readerColumn = [];
            let rightColumns = [];
            let allRightColumn = [];
            for (i in permissions)
            {
                let permission = permissions[i];
                switch (permission.type)
                {
                    case 'READ':
                        readerColumn = Ametys.plugins.coreui.profiles.ProfilesGridHelper.getProfilesColumns([{id: 'READER', label: permission.label}], renderer, scope);
                        readerColumn[0].dataIndex = permission.key;
                        break;
                    case 'ALL_RIGHTS':
                        allRightColumn = Ametys.plugins.coreui.profiles.ProfilesGridHelper.getProfilesColumns([{id: permission.key, label: permission.label}], renderer, scope);
                        allRightColumn[0].cls = 'a-column-header-all';
                        allRightColumn[0].tdCls = 'a-grid-cell-all';
    
                        break;
                    case 'PROFILE':
                        profileColumns.push({id: permission.key, label: permission.label, rights: permission.rights});
                        break;
                    case 'RIGHT':
                        rightColumns.push({id: permission.key, label: permission.label, rights: permission.rights});
                        break;
                    default:
                        // ignore
                }
            }
            rightColumns = Ametys.plugins.coreui.profiles.ProfilesGridHelper.getProfilesColumns(rightColumns, renderer, scope);
            rightColumns = allRightColumn.concat(rightColumns);
            profileColumns = Ametys.plugins.coreui.profiles.ProfilesGridHelper.getProfilesColumns(profileColumns, renderer, scope);
            profileColumns = readerColumn.concat(profileColumns);
            
            if (profileColumns.length > 0 && rightColumns.length > 0)
            {
                profileColumns[profileColumns.length - 1].tdCls = profileColumns[profileColumns.length - 1].tdCls + " add-border";
            }
            
            return [{
                text: "{{i18n PLUGINS_CORE_UI_TOOL_USER_PROFILES_PROFILES_COLUMN_LABEL}}",
                tooltip: "{{i18n PLUGINS_CORE_UI_TOOL_USER_PROFILES_PROFILES_COLUMN_DESCRIPTION}}",
                sortable: false,
                align: 'center',
                cls : profileColumns.length > 0 && rightColumns.length > 0 ? 'add-border' : '',
                columns: profileColumns,
                sealed: true
            }, {
                text: "{{i18n PLUGINS_CORE_UI_TOOL_USER_PROFILES_RIGHTS_COLUMN_LABEL}}",
                tooltip: "{{i18n PLUGINS_CORE_UI_TOOL_USER_PROFILES_RIGHTS_COLUMN_DESCRIPTION}}",
                sortable: false,
                align: 'center',
                columns: rightColumns,
                sealed: true
            }];
        },
    
    /**
     * Compute the tool tip for a record
     * @param {Object[]} accessExplanations A list of access explanations
     * @return {String} The HTML string to be rendered
     */
    computeExplanation: function (accessExplanations)
    {
        var tooltip = '';
        
        for (i in accessExplanations)
        {
            var explanation = accessExplanations[i];
            var glyph = Ametys.plugins.coreui.profiles.ProfilesGridHelper._computeExplanationGlyph(explanation.accessResult);
            var label = explanation.explanation;
            tooltip += `<div><i class="icon ${glyph}"></i>${label}</div>`;
        }
        
        return tooltip;
    },
    
    /**
     * @private
     * Gets a list of css classes based on an access result.
     * @param accessResult a string representing an access result
     * @return {String} The HTML string to be rendered
     */
    _computeExplanationGlyph: function (accessResult)
    {
        switch (accessResult) 
        {
            case Ametys.plugins.coreui.rights.AccessResult.ANONYMOUS_ALLOWED:
            case Ametys.plugins.coreui.rights.AccessResult.ANY_CONNECTED_ALLOWED:
            case Ametys.plugins.coreui.rights.AccessResult.GROUP_ALLOWED:
            case Ametys.plugins.coreui.rights.AccessResult.USER_ALLOWED:
                return "ametysicon-check-1 allowed";
            case Ametys.plugins.coreui.rights.AccessResult.ANONYMOUS_DENIED:
            case Ametys.plugins.coreui.rights.AccessResult.ANY_CONNECTED_DENIED:
            case Ametys.plugins.coreui.rights.AccessResult.GROUP_DENIED:
            case Ametys.plugins.coreui.rights.AccessResult.USER_DENIED:
                return "ametysicon-cross-1 denied";
            case Ametys.plugins.coreui.rights.AccessResult.UNKNOWN:
            default:
                // Undetermined
                return "ametysicon-minus-symbol unknown";
        }
    },

    }
});
