/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * Abstract class representing a convertor able to convert a message target
 * in a right context.
 * To create a new convertor, extends this class and implements its template methods.
 */
Ext.define('Ametys.plugins.coreui.rights.AbstractTargetToContextConvertor', {
    /**
     * @private
     * @property {String} _serverId The id of this component on the server side
     */
    
    /**
     * @protected
     * This methods should return the server-side role of the component to call.
     * @return {String} The component role
     */
    getServerId: function()
    {
        return this._serverId;
    },
    
    /**
     * @protected
     * This methods set the server-side role of the component to call.
     * @param {String} serverId The component role
     */
    setServerId: function(serverId)
    {
        this._serverId = serverId;
    },
    
    /**
     * Indicate if the convertor support a given target
     * @param {Ametys.message.MessageTarget} target a message target
     * @return {boolean} true if the convertor is able to retrieve a right context
     * from this target
     * @template
     */
    isSupportedTarget: function(target)
    {
        throw new Error("This method is not implemented in " + this.getName());
    },
    
    /**
     * Compares two targets and returns true if the two targets are equals.
     * 
     * This method should only be called with supported targets.
     * 
     * The default implementation check that the two targets have the same parameter "id""
     * @return true if the two targets are the same.
     */
    areSameTargets: function(target1, target2)
    {
        if (target1.getParameters().id && target2.getParameters().id)
        {
            return target1.getParameters().id == target2.getParameters().id;
        }
        return false;
    },
    
    /**
     * Convert a message target into a right context.
     * @param {Ametys.message.MessageTarget} target a message target
     * @return {Object} a right context
     * @template
     */
    convert: function(target)
    {
        throw new Error("This method is not implemented in " + this.getName());
    },
    
    /**
     * Get info about the targeted context
     * @param {Ametys.message.MessageTarget} target a message target
     * @return {Object} a label and type properties are mandatory
     * @template
     */
    getContextInfo:function(target)
    {
        throw new Error("This method is not implemented in " + this.getName());
    }
});

