/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton helper class to open a dialog box for asking information before launching the execution of an asynchronous script.
 */
Ext.define('Ametys.plugins.coreui.script.AsyncScriptDialog', {
    singleton: true,
    
    /**
     * @private
     * @property {Ametys.window.DialogBox} _dialog The dialog box for asking information before launching the execution of the asynchronous script.
     */
    /**
     * @private
     * @property {Function} _serverCallFn The function to call when the form is valid AND the 'Ok' button is clicked
     */
    /**
     * @private
     * @property {Object} _scope The scope for the function
     */
    
    /**
     * Opens the dialog box.
     * @param {Object} config The configuration
     * @param {Function} config.serverCallFn The function to call when the form is valid AND the 'Ok' button is clicked
     * @param {Object} config.serverCallFn.params The only function argument: all parameters
     * @param {String} config.serverCallFn.params.recipient The recipient filled by the user in this dialog box
     * @param {Object} config.scope The scope for the function
     */
    open: function(config)
    {
        this._serverCallFn = config.serverCallFn;
        this._scope = config.scope;
        if (this._dialog == null)
        {
            this._dialog = this._createDialogBox();
        }

        this._initForm();
        this._dialog.show();
    },
    
    /**
     * Creates the dialog box
     * @return {Ametys.window.DialogBox} The dialog box
     * @private
     */
    _createDialogBox: function()
    {
        var formPanel = Ext.create('Ext.form.Panel', {
            itemId: 'form',
            border: false,
                     
            scrollable: false,
           
            defaults: {
                cls: 'ametys',
                labelSeparator: '',
                labelAlign: 'right',
                labelWidth: 60,
                anchor: '90%'
            },
            items: [
                {
                    xtype: 'component',
                    id: 'inputRecipientHint',
                    html: "{{i18n PLUGINS_CORE_UI_SCRIPT_ASYNC_EXECUTE_DIALOG_HINT}}",
                    cls: 'text'
                },
                {
                    xtype: 'textfield',
                    id: 'recipient',
                    name: 'recipient',
                    fieldLabel: "{{i18n PLUGINS_CORE_UI_SCRIPT_ASYNC_EXECUTE_DIALOG_RECIPIENT_LABEL}}",
                    ametysDescription: "{{i18n PLUGINS_CORE_UI_SCRIPT_ASYNC_EXECUTE_DIALOG_RECIPIENT_DESC}}",
                    allowBlank: false,
                    emptyText: "{{i18n PLUGINS_CORE_UI_SCRIPT_ASYNC_EXECUTE_DIALOG_RECIPIENT_EMPTY_TEXT}}"
                }
            ]
        });
        
        var dialog = Ext.create('Ametys.window.DialogBox', {
            layout: 'fit',
            
            title: "{{i18n PLUGINS_CORE_UI_SCRIPT_ASYNC_EXECUTE_DIALOG_TITLE}}",
            iconCls: 'ametysicon-play124 decorator-ametysicon-arrow-circle-right-double',
            
            width: 480,
            scrollable: false,
            
            items: [formPanel],
            
            closeAction: 'hide',
            buttons: [{
                text: "{{i18n PLUGINS_CORE_UI_SCRIPT_ASYNC_EXECUTE_DIALOG_BTN_OK}}",
                handler: this._ok,
                scope: this
            }, {
                text: "{{i18n PLUGINS_CORE_UI_SCRIPT_ASYNC_EXECUTE_DIALOG_BTN_CANCEL}}",
                handler: function() {
                    this._dialog.close();
                },
                scope: this
            }]
        });
        
        return dialog;
    },
    
    /**
     * Initializes the form.
     * @private
     */
    _initForm: function()
    {
        var user = Ametys.getAppParameter('user');
        var form = this._dialog.items.getByKey('form').getForm();
        form.reset();
        form.findField('recipient').setValue(user && user.email || '' );
    },
    
    /**
     * The action to perform when the user clicks on the OK button from the dialog box.
     * @private
     */
    _ok: function()
    {
        var form = this._dialog.items.getByKey('form').getForm();

        var params = form.getValues();
            
        if (form.isValid())
        {
            this._dialog.close();
            this._serverCallFn.call(this._scope, params);
        }
    }
});
