/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/**
 * This tool displays the list of users
 * @private
 */
Ext.define('Ametys.plugins.coreui.users.UsersGroupsTool', {
    extend: 'Ametys.tool.Tool',
    
    /**
     * @private
     * @property {String} _login The current user's login
     */
    /**
     * @private
     * @property {String} __populationId The current user's population id
     */
    
    /**
     * @property {Ext.data.Store} _store The grid store
     * @private
     */

    getMBSelectionInteraction: function() 
    {
        return Ametys.tool.Tool.MB_TYPE_ACTIVE;
    },

    setParams: function(params)
    {
        this.callParent(arguments);
        
        this._login = params.login;
        this._populationId = params.populationId;
        Ametys.plugins.core.users.UsersDAO.getUser([this._login, this._populationId], this._getUserCb, {scope: this, ignoreCallbackOnError: false});
        
        this._store.getProxy().setExtraParams({
            user: {login: this._login, populationId: this._populationId}
        });
        
        this.refresh();
    },
    
    /**
     * @private
     * The callback for getuser
     * @param {Object} user the user's properties
     */
    _getUserCb: function(user)
    {
        if (user)
        {
            this.setTitle(new Ext.Template("{{i18n PLUGINS_CORE_UI_TOOL_USERSGROUPS_LABEL}}").apply([user.fullname]));
            this.setDescription(new Ext.Template("{{i18n PLUGINS_CORE_UI_TOOL_USERSGROUPS_DESCRIPTION}}").apply([user.login, user.populationLabel]))
        }
        else
        {
            this.close();
            Ametys.notify({
                title: "{{i18n PLUGINS_CORE_UI_TOOL_USERSGROUPS_AUTOCLOSE_TITLE}}",
                description: "{{i18n PLUGINS_CORE_UI_TOOL_USERSGROUPS_AUTOCLOSE_DESCRIPTION}}",
                type: "warn"
            });
            this.getLogger().error("Unknown user " + this._login + "#" + this._populationId);
        }
    },
    
    createPanel: function()
    {
        this._store = Ext.create('Ext.data.Store', {
            model: 'Ametys.plugins.coreui.groups.GroupsTool.Group',
            
            proxy: {
                type: 'ametys',
                role: 'org.ametys.plugins.core.group.GroupDAO',
                methodName: 'getUsersGroup',
                methodArguments: ['user'],

                reader: {
                    type: 'json'
                }
            },
            
            sorters: {property: 'label', direction: 'ASC'}
        });

        return Ext.create('Ext.grid.Panel', {
            store: this._store,
            stateful: true,
            stateId: this.self.getName() + "$grid",
            columns: [
                {header: "{{i18n PLUGINS_CORE_UI_TOOL_GROUPS_LABEL}}", width: 150, dataIndex: 'label', renderer: this._renderGroupName, hideable: false},
                {header: "{{i18n PLUGINS_CORE_UI_TOOL_GROUPS_ID}}", width: 100, dataIndex: 'groupId', hideable: false},
                {header: "{{i18n PLUGINS_CORE_UI_TOOL_GROUPS_DIRECTORY}}", flex: 1, dataIndex: 'groupDirectoryLabel'}
            ]
        });
    },
    
    /**
     * @private
     * Function to render the name of a group
     * @param {Object} value The data value
     * @param {Object} metaData A collection of data about the current cell
     * @param {Ext.data.Model} record The record
     * @return {String} The html value to render.
     */
    _renderGroupName: function(value, metaData, record)
    {
        return '<span class="a-grid-glyph ametysicon-multiple25"></span>' + value;
    },


    refresh: function()
    {
        this._store.load();
    },
    
    sendCurrentSelection: function()
    {
        Ext.create('Ametys.message.Message', {
            type: Ametys.message.Message.SELECTION_CHANGED,
            targets: []
        });
    }
});
