/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

// Move the profile image user pref from user preferences to user global preferences
const IOUtils = Java.type("org.apache.commons.io.IOUtils");
const ConnectionHelper = Java.type("org.ametys.core.datasource.ConnectionHelper");
const UserIdentity = Java.type('org.ametys.core.user.UserIdentity');
const UserPrefsHandler = Java.type('org.ametys.core.userpref.UserPrefsHandler');
const SQLDatabaseTypeExtensionPoint = Java.type('org.ametys.core.datasource.dbtype.SQLDatabaseTypeExtensionPoint');
const UserPreferencesManager = Java.type('org.ametys.core.userpref.UserPreferencesManager');
const SAXParser = Java.type('org.apache.excalibur.xml.sax.SAXParser');
const SQLDataSourceManager = Java.type("org.ametys.core.datasource.SQLDataSourceManager");
const SQLScriptHelper = Java.type("org.ametys.core.script.SQLScriptHelper");
const _userPrefsManager = Ametys.serviceManager.lookup(UserPreferencesManager.ROLE);
const _sqlDatabaseTypeExtensionPoint = Ametys.serviceManager.lookup(SQLDatabaseTypeExtensionPoint.ROLE);
const _saxParser = Ametys.serviceManager.lookup(SAXParser.ROLE);

// Get the datasource from config
let userPrefsDataSource = org.ametys.runtime.config.Config.getInstance().getValue('runtime.usersprefs.datasource');
if (userPrefsDataSource != null)
{
    // Get database's type
    let connection = ConnectionHelper.getConnection(userPrefsDataSource);
    let dbType = ConnectionHelper.getDatabaseType(connection);
    ConnectionHelper.cleanup(connection);
    
    // Create UserGlobalPreferences table in user pref datasource
    createUserGlobalPreferencesTable(userPrefsDataSource, dbType);
    // Copy the profile user pref from UserPreferences to UserGlobalPreferences for user pref datasource
    copyProfileUserPref(userPrefsDataSource, dbType);
}

let internalDataSource = SQLDataSourceManager.AMETYS_INTERNAL_DATASOURCE_ID;
// Get database's connection and type
let internalConnection = ConnectionHelper.getInternalSQLDataSourceConnection();
let internalDbType = ConnectionHelper.getDatabaseType(internalConnection);

// If table UserPreferences exists in internal database, copy its profiles to the new UserGlobalPreferences table
if (!SQLScriptHelper.tableExists(internalConnection , "UserGlobalPreferences"))
{
    // Create UserGlobalPreferences table in internal datasource
    createUserGlobalPreferencesTable(internalDataSource, internalDbType);
}

ConnectionHelper.cleanup(internalConnection);

function createUserGlobalPreferencesTable(dataSource, dbType)
{
    // Create UserGlobalPreferences database
    let creationQuery;
    
    switch(dbType)
    {
        case 'derby':
            creationQuery = "CREATE TABLE \"UserGlobalPreferences\"(login varchar(64) NOT NULL, population varchar(200) NOT NULL, pref_profile_image BLOB, pref_timezone varchar(128), pref_language varchar(64), PRIMARY KEY(login, population))";
             break;
        case 'hsqldb':
            creationQuery = "CREATE CACHED TABLE \"UserGlobalPreferences\" (login varchar(64) NOT NULL,population varchar(200) NOT NULL, pref_profile_image BLOB, pref_timezone varchar(128), pref_language varchar(64), PRIMARY KEY(login, population));";
            break;
        case 'oracle':
            creationQuery = "CREATE TABLE \"UserGlobalPreferences\"(login varchar(64) NOT NULL, population varchar(200) NOT NULL, pref_profile_image BLOB, pref_timezone varchar(128), pref_language varchar(64), PRIMARY KEY(login, population))";
            break;
        case 'postgresql':
            creationQuery = "CREATE TABLE \"UserGlobalPreferences\"(login varchar(64) NOT NULL, population varchar(200) NOT NULL, pref_profile_image bytea, pref_timezone varchar(128), pref_language varchar(64), PRIMARY KEY(login, population)); COMMIT;"
            break;
        case 'mysql':
            creationQuery = "CREATE TABLE IF NOT EXISTS `UserGlobalPreferences`(login varchar(64) BINARY NOT NULL, population varchar(200) BINARY NOT NULL, pref_profile_image LONGBLOB, pref_timezone varchar(128), pref_language varchar(64), PRIMARY KEY(login, population)) ENGINE=innodb CHARACTER SET utf8;";
            break;
    }

    // Create user global preferences table
    SQL.update(creationQuery, dataSource);
}

function copyProfileUserPref(dataSource, dbType)
{
    // Get the profile image user preference from UserPreferences
    rs = SQL.query("SELECT login, population, data FROM UserPreferences WHERE Context = '/profile'", dataSource);
    
    let preferencesToAdd = [];
    while (rs.next())
    {
        let login = rs.getString(1);
        let population = rs.getString(2);
    
        let dataIs;
        try
        {
            dataIs = _sqlDatabaseTypeExtensionPoint.getBlob(dbType, rs, "data");
            
            // Parse the xml data and keep only the JSON as String
            let prefs = {};
            let handler = new UserPrefsHandler(prefs);
            _saxParser.parse(new org.xml.sax.InputSource(dataIs), handler);
            
            let dataAsString = prefs["profile-image"];
            
            preferencesToAdd.push({"userIdentity" : new UserIdentity(login, population), "data": dataAsString});
        }
        finally 
        {
            IOUtils.closeQuietly(dataIs);
        }
    }
    
    // Delete values from UserPreferences
    SQL.update("DELETE FROM UserPreferences WHERE Context = '/profile'", dataSource);
    
    // Insert values in UserGlobalPreferences
    for (let preferenceToAdd of preferencesToAdd)
    {
        _userPrefsManager.addUserPreference(preferenceToAdd["userIdentity"], "", {}, "profile-image", preferenceToAdd["data"]);
    }
}
