/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.cocoon;

import java.util.Collections;
import java.util.Map;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.environment.Redirector;

import com.opensymphony.workflow.InvalidInputException;
import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.WorkflowException;

/**
 * Action for initializing a workflow.<p>
 * The following parameters are supported:
 * <dl>
 *  <dt>workflowName
 *  <dd>the name of the workflow to instantiate
 *  <dt>actionId
 *  <dd>the id of the action to fire
 * </dl>
 * {@link Map} returned contains the workflow id on successful creation
 * (<code>workflowId</code> key) and throw an exception on error.
 */
public class InitializeWorkflowAction extends AbstractWorkflowAction
{
    @Override
    protected Map _act(Redirector redirector, Map objectModel, String source, Parameters parameters, int actionId, Map inputs) throws InvalidInputException, WorkflowException
    {
        Workflow workflow = _workflowProvider.getGenericWorkflow();
        long workflowId = workflow.initialize(_getWorkflowName(objectModel, source, parameters), actionId, inputs);
        return _getActionResult(redirector, objectModel, source, parameters, workflowId);
    }

    /**
     * Provides the workflow name to use.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @return the workflow name.
     */
    protected String _getWorkflowName(Map objectModel, String source, Parameters parameters)
    {
        try
        {
            return parameters.getParameter("workflowName");
        }
        catch (ParameterException e)
        {
            throw new RuntimeException("Missing parameter workflowName", e);
        }
    }
    
    /**
     * Provides the action result after successful workflow instance
     * creation.<p>
     * Default implementation provides a singleton {@link Map} with
     * <code>workflowId</code> key and workflow id value.
     * @param redirector the redirector.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @param workflowId the workflow instance id.
     * @return the action result.
     */
    protected Map _getActionResult(Redirector redirector, Map objectModel, String source, Parameters parameters, long workflowId)
    {
        return Collections.singletonMap("workflowId", workflowId);
    }
    
    @Override
    protected String _getExceptionContext(Map objectModel, String source, Parameters parameters)
    {
        return String.format("for the workflow of name: '%s'", _getWorkflowName(objectModel, source, parameters));
    }
}
