/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.component;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.EnhancedCondition;
import org.ametys.plugins.workflow.repository.WorkflowAwareAmetysObject;
import org.ametys.plugins.workflow.support.WorkflowElementDefinitionHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.spi.Step;

/**
 * OSWorkflow condition to test the current step of an Ametys object
 * The following configuration have to be used:<br>
 * 
 * <pre>
 * &lt;condition type="avalon"&gt;
 * &nbsp;&nbsp;&lt;arg name="role"&gt;org.ametys.plugins.workflow.component.AmetysObjectCurrentStepCondition&lt;/arg&gt;
 * &nbsp;&nbsp;&lt;arg name="object-key"&gt;content&lt;/arg&gt;
 * &nbsp;&nbsp;&lt;arg name="step"&gt;2&lt;/arg&gt;
 * &lt;/condition&gt;
 * </pre>
 *
 */
public class AmetysObjectCurrentStepCondition extends AbstractWorkflowComponent implements EnhancedCondition
{
    private WorkflowProvider _workflowProvider;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
    }
    
    public boolean passesCondition(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        int step = Integer.parseInt((String) args.get("step"));
        
        WorkflowAwareAmetysObject ametysObject = (WorkflowAwareAmetysObject) transientVars.get(_getAmetysObjectKey(args));
        if (ametysObject == null)
        {
            throw new WorkflowException("Unable to retrieve the object");
        }
        
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(ametysObject, true);
        
        long workflowId = ametysObject.getWorkflowId();
        
        List<Step> currentSteps = workflow.getCurrentSteps(workflowId);
        for (Step currentStep : currentSteps)
        {
            if (currentStep.getStepId() == step)
            {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get the key to get the {@link WorkflowAwareAmetysObject} into the transient vars.
     * @param args Arguments of the condition
     * @return A key to get the {@link WorkflowAwareAmetysObject}
     */
    protected Object _getAmetysObjectKey(Map args)
    {
        return args.get("object-key");
    }

    @Override
    public List<WorkflowArgument> getArguments()
    {
        return List.of(
            WorkflowElementDefinitionHelper.getElementDefinition(
                "step",
                new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_AMETYS_OBJECT_CURRENT_STEP_CONDITION_ARGUMENT_STEP_LABEL"),
                new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_AMETYS_OBJECT_CURRENT_STEP_CONDITION_ARGUMENT_STEP_DESCRIPTION"),
                true,
                false
            ),
             WorkflowElementDefinitionHelper.getElementDefinition(
                "object-key",
                new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_AMETYS_OBJECT_CURRENT_STEP_CONDITION_ARGUMENT_OBJECT_KEY_LABEL"),
                new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_AMETYS_OBJECT_CURRENT_STEP_CONDITION_ARGUMENT_OBJECT_KEY_DESCRIPTION"),
                true,
                false
            )
        );
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.workflow", "UITOOL_WORKFLOW_EDITOR_AMETYS_OBJECT_CURRENT_STEP_CONDITION_LABEL");
    }
    
    @Override
    public I18nizableText getFullLabel(Map<String, String> argumentsValues)
    {
        String stepId = argumentsValues.get("step");
        return new I18nizableText("plugin.workflow", "UITOOL_WORKFLOW_EDITOR_AMETYS_OBJECT_CURRENT_STEP_CONDITION_DESCRIPTION", List.of(stepId));
    }
}
