/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.workflow.component.WorkflowLanguageManager;
import org.ametys.plugins.workflow.support.I18nHelper;
import org.ametys.plugins.workflow.support.WorflowRightHelper;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowSessionHelper;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.DescriptorFactory;
import com.opensymphony.workflow.loader.ResultDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * The workflow action DAO
 */
public class WorkflowTransitionDAO implements Component, Serviceable
{
    /** The component's role */
    public static final String ROLE =  WorkflowTransitionDAO.class.getName();
    
    /** The default label for actions */
    public static final I18nizableText DEFAULT_ACTION_NAME = new I18nizableText("plugin.workflow", "PLUGIN_WORKFLOW_DEFAULT_ACTION_LABEL");    
    
    /** Default path for svg action icons */
    private static final String __DEFAULT_SVG_ACTION_ICON_PATH = "plugin:cms://resources/img/history/workflow/action_0_16.png";
    
    /** Default path for node action icons */
    private static final String __DEFAULT_ACTION_ICON_PATH = "/plugins/cms/resources/img/history/workflow/action_0_16.png";
    
    /** The workflow session helper */
    protected WorkflowSessionHelper _workflowSessionHelper;
    
    /** The workflow right helper */
    protected WorflowRightHelper _workflowRightHelper;
    
    /** The Workflow Language Manager */
    protected WorkflowLanguageManager _workflowLanguageManager;
    
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    /** The workflow step DAO */
    protected WorkflowStepDAO _workflowStepDAO;
    
    /** The helper for i18n translations and catalogs */
    protected I18nHelper _i18nHelper;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The context */
    protected Context _context;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _workflowSessionHelper = (WorkflowSessionHelper) smanager.lookup(WorkflowSessionHelper.ROLE);
        _workflowHelper = (WorkflowHelper) smanager.lookup(WorkflowHelper.ROLE);
        _workflowRightHelper = (WorflowRightHelper) smanager.lookup(WorflowRightHelper.ROLE);
        _i18nHelper = (I18nHelper) smanager.lookup(I18nHelper.ROLE);
        _workflowLanguageManager = (WorkflowLanguageManager) smanager.lookup(WorkflowLanguageManager.ROLE);
        _workflowStepDAO = (WorkflowStepDAO) smanager.lookup(WorkflowStepDAO.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    /**
     * Get the transition infos to initialize creation/edition form fields
     * @param workflowName the current workflow name 
     * @param transitionId the current transition's id, can be null
     * @return the transition values and a list of taken ids
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getTransitionInfos(String workflowName, Integer transitionId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        Map<String, Object> transitionInfos = new HashMap<>();
        Set<Integer> transitionIds = _workflowHelper.getAllActions(workflowDescriptor);
        I18nizableText labelKey = DEFAULT_ACTION_NAME;
        if (transitionId != null)
        {
            transitionIds.remove(transitionId);
            transitionInfos.put("id", transitionId);
            ActionDescriptor action = workflowDescriptor.getAction(transitionId);
            labelKey = getActionLabel(action); 
            
            transitionInfos.put("finalStep", action.getUnconditionalResult().getStep());
        }
        transitionInfos.put("ids", transitionIds);
        
        Map<String, String> translations = _workflowSessionHelper.getTranslation(workflowName, labelKey);
        if (translations == null)
        {
            translations = new HashMap<>();
            translations.put(_workflowLanguageManager.getCurrentLanguage(), _i18nHelper.translateKey(workflowDescriptor.getName(), labelKey, DEFAULT_ACTION_NAME));
        }
        transitionInfos.put("labels", translations);
        
        return transitionInfos;
    }
    
    /**
     * Get a set of transitions already defined in current workflow beside initialActions
     * @param workflowName the current workflow name 
     * @param parentStepId the current selected step
     * @return a set containing maps of actions properties
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Set<Map<String, Object>> getAvailableActions(String workflowName, Integer parentStepId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkReadRight(workflowDescriptor);
        
        
        Set<Map<String, Object>> availableActions = new HashSet<>();
        if (0 != parentStepId) // generic actions can't be used as  initial actions
        {
            StepDescriptor parentStep = workflowDescriptor.getStep(parentStepId);
            List<ActionDescriptor> actions = parentStep.getActions();
            
            List<StepDescriptor> steps = workflowDescriptor.getSteps();
            for (StepDescriptor step: steps)
            {
                List<ActionDescriptor> outgoingActions = step.getActions();
                for (ActionDescriptor outgoingAction: outgoingActions)
                {
                    if (!actions.contains(outgoingAction))
                    {
                        availableActions.add(_getActionInfos(workflowName, outgoingAction));
                    }
                }
            }
        }
        return availableActions;
    }

    private Map<String, Object> _getActionInfos(String workflowName, ActionDescriptor action)
    {
        Map<String, Object> actionInfos = new HashMap<>();
        int actionId = action.getId();
        actionInfos.put("id", actionId);
        actionInfos.put("label", getActionLabel(workflowName, action) + " (" + actionId + ")");
        return actionInfos;
    }
    
    /**
     * Create a new transition
     * @param workflowName the current workflow name
     * @param parentStepId the parent step id
     * @param transitionId the id for the transition to create
     * @param labels the multilinguals labels for the transition
     * @param finalStepId the id for the transition's unconditional result
     * @return a map with error message or with transition's id if succesfull 
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> createTransition(String workflowName, Integer parentStepId, int transitionId, Map<String, String> labels, int finalStepId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        if (_workflowHelper.getAllActions(workflowDescriptor).contains(transitionId))
        {
            return Map.of("message", "duplicate-id");
        }
        
        DescriptorFactory factory = new DescriptorFactory();
        ActionDescriptor action = factory.createActionDescriptor();
        action.setId(transitionId);
        I18nizableText actionNameI18nKey = _i18nHelper.generateI18nKey(workflowName, "ACTION", transitionId);
        action.setName(actionNameI18nKey.toString());
        ResultDescriptor finalStep = factory.createResultDescriptor();
        finalStep.setStep(finalStepId);
        action.setUnconditionalResult(finalStep);
        
        if (isInitialStep(parentStepId))
        {
            workflowDescriptor.addInitialAction(action);
        }
        else
        {
            StepDescriptor stepDescriptor = workflowDescriptor.getStep(parentStepId);
            stepDescriptor.getActions().add(action);
        }
        
        _workflowSessionHelper.updateTranslations(workflowName, actionNameI18nKey, labels);
        _generateActionDescription(workflowName, actionNameI18nKey, labels);
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getActionProperties(workflowDescriptor, action, parentStepId);
    }

    private Map<String, Object> _getActionProperties(WorkflowDescriptor workflowDescriptor, ActionDescriptor action, Integer stepId)
    {
        Map<String, Object> results = new HashMap<>();
        results.put("actionId", action.getId());
        results.put("actionLabels", getActionLabel(workflowDescriptor.getName(), action));
        results.put("stepId", stepId);
        results.put("stepLabel", _workflowStepDAO.getStepLabel(workflowDescriptor, stepId));
        results.put("workflowId", workflowDescriptor.getName());
        
        return results;
    }

    private boolean isInitialStep(Integer stepIdToInt)
    {
        return stepIdToInt == 0;
    }
    
    /**
     * Add an existing transition to current step
     * @param workflowName the current workflow name
     * @param parentStepId the current selected step id
     * @param transitionIds a list of ids corresponding to the transitions to add to current step
     * @return the first transition id and its parent sted id
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addTransitions(String workflowName, Integer parentStepId, List<Integer> transitionIds)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        Map<String, Object> results = new HashMap<>();
        
        if (parentStepId != 0)
        {
            StepDescriptor step = workflowDescriptor.getStep(parentStepId);
            for (Integer id : transitionIds)
            {
                ActionDescriptor action = _getAction(workflowDescriptor, id);
                step.getActions().add(action);
                if (!action.isCommon())
                {
                    _updateWorkflowCommonAction(workflowDescriptor, id, action);
                }
                else
                {
                    step.getCommonActions().add(id);
                }
            }
            _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
            results = _getActionProperties(workflowDescriptor, workflowDescriptor.getAction(transitionIds.get(0)), parentStepId);
        }
        else
        {
            results.put("message", "initial-step");
        }
        
        return results;
    }
    
    private ActionDescriptor _getAction(WorkflowDescriptor workflowDescriptor, Integer actionId)
    {
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        if (action == null)
        {
            Map<Integer, ActionDescriptor> commonActions = workflowDescriptor.getCommonActions();
            action = commonActions.get(actionId);
        }
        return action;
    }
    
    /**
     * Set action as 'common' in steps using it
     * @param workflowDescriptor the current workflow
     * @param actionId id of current action
     * @param action the action
     */
    protected void _updateWorkflowCommonAction(WorkflowDescriptor workflowDescriptor, int actionId, ActionDescriptor action)
    {
        List<StepDescriptor> stepsToUpdate = new ArrayList<>();
        List<StepDescriptor> steps = workflowDescriptor.getSteps();
        
        //remove individual action from steps 
        for (StepDescriptor step : steps)
        {
            if (step.getAction(actionId) != null)
            {
                step.getActions().remove(action);
                stepsToUpdate.add(step);
            }
        }
        //set action as common in workflow
        workflowDescriptor.addCommonAction(action);
        
        //put back action in steps as common action
        for (StepDescriptor step : stepsToUpdate)
        {
            step.getCommonActions().add(actionId);
            step.getActions().add(action);
        }
    }
    
    /**
     * Rename the transition
     * @param workflowName the workflow name
     * @param stepId id of selected step
     * @param actionId the transition's id
     * @param newMainLabel the new label in current language
     * @return  a map with error message or with transition's infos if succesfull 
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editTransitionLabel(String workflowName, Integer stepId, Integer actionId, String newMainLabel)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        _updateActionName(workflowName, action);
        I18nizableText actionKey = getActionLabel(action);
        _workflowSessionHelper.updateTranslations(workflowName, actionKey, Map.of(_workflowLanguageManager.getCurrentLanguage(), newMainLabel));
        
        _generateActionDescription(workflowName, actionKey, Map.of(_workflowLanguageManager.getCurrentLanguage(), newMainLabel));
        
        return _getActionProperties(workflowDescriptor, action, stepId);
    }

    private void _updateActionName(String workflowName, ActionDescriptor action)
    {
        String defaultCatalog = _workflowHelper.getWorkflowCatalog(workflowName);
        I18nizableText actionKey = getActionLabel(action);
        
        if (!defaultCatalog.equals(actionKey.getCatalogue()))
        {
            String newName = new I18nizableText(defaultCatalog, actionKey.getKey()).toString();
            action.setName(newName);
        }
    }
    
    /**
     * Edit the transition
     * @param workflowName the workflow name
     * @param stepId id of selected step
     * @param oldId the transition's former id
     * @param newId the transition's new id
     * @param labels the transition's multilingual labels
     * @param finalStepId the transition's unconditional result id
     * @return a map with error message or with transition's infos if succesfull 
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editTransition(String workflowName, Integer stepId, Integer oldId, Integer newId, Map<String, String> labels, Integer finalStepId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(oldId);
        
        if (!newId.equals(oldId))
        {
            if (_workflowHelper.getAllActions(workflowDescriptor).contains(newId))
            {
                return Map.of("message", "duplicate-id");
            }
            if (action.isCommon())
            {
                //Don't change edit order below: first update steps then update action
                workflowDescriptor.getCommonActions().remove(oldId, action);
                List<StepDescriptor> steps = workflowDescriptor.getSteps();
                for (StepDescriptor step: steps)
                {
                    if (step.getAction(oldId) != null)
                    {
                        List<Integer> commonActions = step.getCommonActions();
                        commonActions.remove(commonActions.indexOf(oldId));
                        commonActions.add(newId);
                    }
                }
                action.setId(newId);
                workflowDescriptor.getCommonActions().put(newId, action);
            }
            else
            {
                action.setId(newId);
            }
        }
        
        DescriptorFactory factory = new DescriptorFactory();
        ResultDescriptor finalStep = factory.createResultDescriptor();
        finalStep.setStep(finalStepId);
        action.setUnconditionalResult(finalStep);
        
        _updateActionName(workflowName, action);
        I18nizableText actionKey = getActionLabel(action);
        _workflowSessionHelper.updateTranslations(workflowName, actionKey, labels);
        
        _generateActionDescription(workflowName, actionKey, labels);
        
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getActionProperties(workflowDescriptor, action, stepId);
    }
    
    private void _generateActionDescription(String workflowName, I18nizableText actionKey, Map<String, String> labels)
    {
        I18nizableText actionDescriptionI18nKey = new I18nizableText(actionKey.getCatalogue(), actionKey.getKey() + "_ACTION_DESCRIPTION");
        
        Map<String, String> descriptionValues = labels.entrySet()
            .stream()
            .collect(Collectors.toMap(
                e -> e.getKey(), 
                e -> _getDesciptionValue(e.getValue(), e.getKey())
            )
        );
        
        _workflowSessionHelper.updateTranslations(workflowName, actionDescriptionI18nKey, descriptionValues);
    }
    
    private String _getDesciptionValue(String label, String lang)
    {
        I18nizableText descriptionPrefixI18nText = new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_CREATE_TRANSITION_DEFAULT_DESRIPTION", List.of(label));
        return _i18nUtils.translate(descriptionPrefixI18nText, lang);
    }
    
    /**
     * Remove transition from step
     * @param workflowName the current workflow name
     * @param parentStepId the parent step id to remove the transition from
     * @param transitionId the id for the transition to remove
     * @return empty map if successfull
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> removeTransition(String workflowName, Integer parentStepId, Integer transitionId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ActionDescriptor actionToRemove = null;
        if (isInitialStep(parentStepId))
        {
            actionToRemove = workflowDescriptor.getInitialAction(transitionId);
            _workflowSessionHelper.removeTranslation(workflowName, new I18nizableText("application", actionToRemove.getName()));
            workflowDescriptor.getInitialActions().remove(actionToRemove);
        }
        else
        {
            StepDescriptor stepDescriptor = workflowDescriptor.getStep(parentStepId);
            actionToRemove = workflowDescriptor.getAction(transitionId);
            stepDescriptor.getActions().remove(actionToRemove);
            if (actionToRemove.isCommon())
            {
                List<Integer> commonActions = stepDescriptor.getCommonActions();
                commonActions.remove(commonActions.indexOf(transitionId));
                _manageCommonAction(workflowDescriptor, actionToRemove);
            }
            else
            {
                I18nizableText actionKey = getActionLabel(actionToRemove);
                _workflowSessionHelper.removeTranslation(workflowName, actionKey); //use actionKey instead of action.getName() in case plugin's name is in the name
            }
        }
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getActionProperties(workflowDescriptor, actionToRemove, parentStepId);
    }
    
    /**
     * Verify that the action is still used by multiple steps. If not anymore, replace the action by a non common copy 
     * @param workflowDescriptor the current workflow
     * @param action the action removed
     */
    protected void _manageCommonAction(WorkflowDescriptor workflowDescriptor, ActionDescriptor action)
    {
        Map<Integer, ActionDescriptor> commonActions = workflowDescriptor.getCommonActions();
        Integer id = action.getId();
        if (commonActions.containsKey(id))
        {
            List<StepDescriptor> steps = workflowDescriptor.getSteps();
            int numberOfUse = _getNumberOfUse(id, steps);
            if (numberOfUse == 1)
            {
                //only way to unset common in action is to replace it with a new copy 
                StepDescriptor parentStep = _getParentStep(steps, id).get();
                List<Integer> stepCommonActions = parentStep.getCommonActions();
                stepCommonActions.remove(stepCommonActions.indexOf(id));
                parentStep.getActions().remove(action);
                workflowDescriptor.getCommonActions().remove(id, action);
                ActionDescriptor actionCopy =  _copyAction(action);
                parentStep.getActions().add(actionCopy);
                _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
            }
        }
    }
    
    private Optional<StepDescriptor> _getParentStep(List<StepDescriptor> steps, int actionId)
    {
        return steps.stream().filter(s -> s.getAction(actionId) != null).findFirst();
    }

    private ActionDescriptor _copyAction(ActionDescriptor actionToCopy)
    {
        DescriptorFactory factory = new DescriptorFactory();
        ActionDescriptor action = factory.createActionDescriptor();
        action.setId(actionToCopy.getId());
        action.setName(actionToCopy.getName());
        action.setUnconditionalResult(actionToCopy.getUnconditionalResult());
        action.getConditionalResults().addAll(actionToCopy.getConditionalResults());
        action.setMetaAttributes(actionToCopy.getMetaAttributes());
        action.setRestriction(actionToCopy.getRestriction());
        action.getPreFunctions().addAll(actionToCopy.getPreFunctions());
        action.getPostFunctions().addAll(actionToCopy.getPostFunctions());
        
        return action;
    }
    
    /**
     * Get the number of steps using the action
     * @param workflowName the current workflow's name
     * @param actionId the current action's id
     * @return the number of steps using the action
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public int getNumberOfUse(String workflowName, Integer actionId)
    {
        WorkflowDescriptor workflow = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        _workflowRightHelper.checkReadRight(workflow);
        
        return _getNumberOfUse(actionId, workflow.getSteps());
    }
    
    private int _getNumberOfUse(Integer actionId, List<StepDescriptor> steps)
    {
        int stepSize = steps.size();
        int numberOfUse = 0;
        int index = 0;
        while (index < stepSize)
        {
            if (steps.get(index).getAction(actionId) != null)
            {
                numberOfUse++;
            }
            index++;
        }
        return numberOfUse;
    }
    
    /**
     * Get the translated action label
     * @param workflowName the workflow's unique name
     * @param action current action
     * @return the action label 
     */
    public String getActionLabel(String workflowName, ActionDescriptor action)
    {
        I18nizableText label = getActionLabel(action);
        return _i18nHelper.translateKey(workflowName, label, DEFAULT_ACTION_NAME);
    }
    
    /**
     * Get the action's icon path 
     * @param workflowName name of current workflow
     * @param action current action
     * @return the icon's path
     */
    public String getActionIconPath(String workflowName, ActionDescriptor action)
    {
        I18nizableText label = getActionLabel(action);
        label = _workflowSessionHelper.getOldLabelKeyIfCloned(workflowName, label); 
        return _workflowHelper.getElementIconPath(label, __DEFAULT_ACTION_ICON_PATH);
    }
    
    /**
     * Get the action's icon path as base 64 for svg's links
     * @param workflowName name of current workflow
     * @param action current action
     * @return the icon's path as base 64
     */
    public String getActionIconPathAsBase64(String workflowName, ActionDescriptor action) 
    {
        I18nizableText label = getActionLabel(action);
        label = _workflowSessionHelper.getOldLabelKeyIfCloned(workflowName, label);
        return _workflowHelper.getElementIconAsBase64(label, __DEFAULT_SVG_ACTION_ICON_PATH);
    }
    
    /**
     * Get the action label as i18n
     * @param action the current action
     * @return the label as i18n
     */
    public I18nizableText getActionLabel(ActionDescriptor action)
    {
        return new I18nizableText("application", action.getName());
    }
}
