/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.readers;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.reading.AbstractReader;
import org.apache.commons.io.IOUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.workflow.dao.WorkflowStepDAO;
import org.ametys.plugins.workflow.dao.WorkflowTransitionDAO;
import org.ametys.plugins.workflow.support.I18nHelper;
import org.ametys.plugins.workflow.support.WorflowRightHelper;
import org.ametys.plugins.workflow.support.WorkflowSessionHelper;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

import net.sourceforge.plantuml.FileFormat;
import net.sourceforge.plantuml.FileFormatOption;
import net.sourceforge.plantuml.SourceStringReader;

/**
 * Abstract class for reading PlantUML SVG
 */
public abstract class AbstractPlantUMLSVGReader extends AbstractReader implements Serviceable
{
    /** The color for step nodes */
    protected static final String __MAIN_STEP_NODE_COLOR = "#e5f8ff";
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    
    /** The i18n workflow helper */
    protected I18nHelper _i18nHelper;
    
    /** The workflow session helper */
    protected WorkflowSessionHelper _workflowSessionHelper;
    
    /** The workflow step DAO */
    protected WorkflowStepDAO _workflowStepDAO;
    
    /** The workflow transition DAO */
    protected WorkflowTransitionDAO _workflowTransitionDAO;
    
    /** The workflow right helper */
    protected WorflowRightHelper _workflowRightHelper;
    
    /** The JSON Utils */
    protected JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _i18nHelper = (I18nHelper) serviceManager.lookup(I18nHelper.ROLE);
        _workflowStepDAO = (WorkflowStepDAO) serviceManager.lookup(WorkflowStepDAO.ROLE);
        _workflowTransitionDAO = (WorkflowTransitionDAO) serviceManager.lookup(WorkflowTransitionDAO.ROLE);
        _workflowSessionHelper = (WorkflowSessionHelper) serviceManager.lookup(WorkflowSessionHelper.ROLE);
        _workflowRightHelper = (WorflowRightHelper) serviceManager.lookup(WorflowRightHelper.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public String getMimeType()
    {
        return "image/svg+xml";
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        try
        {
            Request request = ObjectModelHelper.getRequest(objectModel);
            String workflowName = (String) request.get("workflowName");
            _setContextInRequestAttributes(request);
            WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
            if (_workflowRightHelper.canRead(workflowDescriptor))
            {
                _setPlantUMLProperties();
                String content = _getPlantUMLContent(request, workflowDescriptor);
                SourceStringReader reader = new SourceStringReader(content);
                
                try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream())
                {
                    reader.outputImage(byteArrayOutputStream, new FileFormatOption(FileFormat.SVG).withSvgLinkTarget("_self"));
                    
                    String svgContent = byteArrayOutputStream.toString(StandardCharsets.UTF_8);
                    svgContent = svgContent.replaceAll("<svg ", "<svg onclick=\"parent.Ametys.tool.ToolsManager.getTool('uitool-workflow-preview').focus();\" ");
                    
                    IOUtils.write(svgContent, out, StandardCharsets.UTF_8);
                }
            }
        }
        finally
        {
            out.close();
        }
    }
    
    /**
     * Set all context parameters separatly in request
     * @param request the request
     */
    protected void _setContextInRequestAttributes(Request request)
    {
        Map<String, Object> contextAsMap = _jsonUtils.convertJsonToMap(request.getParameter("context.parameters"));
        if (contextAsMap != null)
        {
            for (String name : contextAsMap.keySet())
            {
                request.setAttribute(name, contextAsMap.get(name));
            }
        }
    }

    /**
     * Set common plantUML svg properties
     */
    protected void _setPlantUMLProperties()
    {
        System.setProperty("PLANTUML_ALLOW_JAVASCRIPT_IN_LINK", "true");
        System.setProperty("PLANTUML_LIMIT_SIZE", String.valueOf(Integer.MAX_VALUE));
    }

    /**
     * Get the PlantUML SVG content to read
     * @param request the request
     * @param workflowDescriptor descriptor of current workflow
     * @return The content to read
     */
    protected String _getPlantUMLContent(Request request, WorkflowDescriptor workflowDescriptor)
    {
        StringBuilder content = new StringBuilder("@start" + _getPlantUMLType() + "\n");
        content.append(_getPlantUMLStyle());
        content.append(_getPlantUMLGraphContent(request, workflowDescriptor));
        content.append("@end" + _getPlantUMLType());
        return content.toString();
    }
    
    /**
     * Get the diagram type
     * @return the diagram type
     */
    protected abstract String _getPlantUMLType();

    /**
     * Get plantUML style for current diagram
     * @return the style as string
     */
    protected abstract String _getPlantUMLStyle();

    /**
     * Get the plantUML diagram body
     * @param request the request
     * @param workflowDescriptor descriptor of current workflow
     * @return the diagram body as string
     */
    protected abstract String _getPlantUMLGraphContent(Request request, WorkflowDescriptor workflowDescriptor);

    /**
     * Get js function to send selection
     * @param workflowName unique name of current workflow
     * @param stepId current or incoming step's id
     * @param stepLabel current or incoming step's label
     * @param actionId current action's id, can be null
     * @param actionLabel current action's label, can be null
     * @return the js function
     */
    protected String _getJsFunction(String workflowName, String stepId, String stepLabel, String actionId, String actionLabel)
    {
        String jsParameters = "'" + workflowName + "','" + stepId + "','" + stepLabel + "','";
        jsParameters += actionId != null ? actionId + "','" + actionLabel + "'" : "'";
        return "Ametys.plugins.workflow.UI.UIWorkflowGraph.sendSelection(" + jsParameters + ")";
    }
    
    /**
     * Get the node label for step
     * @param workflowDescriptor current workflow
     * @param stepId current step id
     * @return the node label which is the translated step name and its id
     */
    protected String _getStepNodeLabel(WorkflowDescriptor workflowDescriptor, int stepId)
    {
        return _workflowStepDAO.getStepLabelAsString(workflowDescriptor, stepId, true);
    }

    /**
     * Get the tooltip for a link on step
     * @param step current step
     * @return the tooltip
     */
    protected String _getStepTooltip(StepDescriptor step)
    {
        return _i18nUtils.translate(new I18nizableText("plugin.workflow", "PLUGIN_WORKFLOW_LINK_SEE_STEP"));
    }

    /**
     * Get the action label for node
     * @param workflowName the workflow's unique name
     * @param action current action
     * @return the label as action's name and id
     */
    protected String _getActionLabel(String workflowName, ActionDescriptor action)
    {
        return _workflowTransitionDAO.getActionLabel(workflowName, action) + " (" + action.getId() + ")";
    }
    
    /**
     * Get the tooltip for a link on action
     * @param action current action
     * @return the tooltip
     */
    protected String _getActionTooltip(ActionDescriptor action)
    {
        return _i18nUtils.translate(new I18nizableText("plugin.workflow", "PLUGIN_WORKFLOW_LINK_SEE_TRANSITION"));
    }
    
    /**
     * Get the string with escaped spaces
     * @param stringToEscape a string to process for js function
     * @return the formated string
     */
    protected String _getStringWithEscapedSpace(String stringToEscape)
    {
        return stringToEscape.replaceAll(" ", "&#160;");
    }
}
