/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.readers;

import java.util.List;
import java.util.Set;

import org.apache.cocoon.environment.Request;

import org.ametys.plugins.workflow.dao.WorkflowStepDAO;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Read step mindMap from plantUML
 */
public class PlantUmlMindMapStepSVGReader extends AbstractPlantUMLMindmapSVGReader 
{
    @Override
    protected String _getPlantUMLStyle()
    {
        StringBuilder style = new StringBuilder();
        style.append("<style> \n");
        style.append("mindmapDiagram { \n");
        style.append("boxless { \n");
        style.append("HyperLinkColor #0a7fb2 \n");
        style.append("hyperlinkUnderlineThickness 0 \n"); //workaround until HyperlinkUnderline false is fixed
        style.append("} \n");
        style.append("} \n");
        style.append("</style> \n");
        return style.toString();
    }

    @Override
    protected String _getPlantUMLGraphContent(Request request, WorkflowDescriptor workflowDescriptor)
    {
        StringBuilder content = new StringBuilder();
        String workflowName = workflowDescriptor.getName();
        int stepId = Integer.valueOf((String) request.get("stepId"));
        
        // Graph for central step node
        String nodeContent = _getMindMapNodeContent(
            "+",
            _getStepNodeLabel(workflowDescriptor, stepId),
            _workflowStepDAO.getStepIconPathAsBase64(workflowDescriptor, stepId),
            __MAIN_STEP_NODE_COLOR,
            null,
            null,
            false
        );
        
        content.append(nodeContent);
        
        // Graph for incoming actions
        Set<ActionDescriptor> incomingActions = _workflowStepDAO.getIncomingActions(stepId, workflowDescriptor);
        for (ActionDescriptor action : incomingActions)
        {
            String firstParentStepId = _workflowStepDAO.getFirstParentStepId(stepId, workflowDescriptor.getSteps(), action.getId());
            String firstParentStepLabel = _i18nHelper.translateKey(workflowName, _workflowStepDAO.getStepLabel(workflowDescriptor, Integer.parseInt(firstParentStepId)), WorkflowStepDAO.DEFAULT_STEP_NAME);
            firstParentStepLabel = _getStringWithEscapedSpace(firstParentStepLabel);
            String actionLabel = _workflowTransitionDAO.getActionLabel(workflowName, action);
            actionLabel = _getStringWithEscapedSpace(actionLabel);
            String actionNode = _getMindMapNodeContent(
                "--",
                _getActionLabel(workflowName, action),
                _workflowTransitionDAO.getActionIconPathAsBase64(workflowName, action),
                null,
                _getJsFunction(workflowName, firstParentStepId, firstParentStepLabel, String.valueOf(action.getId()), actionLabel),
                _i18nUtils.translate(new I18nizableText("plugin.workflow", "PLUGIN_WORKFLOW_LINK_SEE_TRANSITION")),
                true
            );
            content.append(actionNode);
        }
        
        // Graph for outgoing action
        List<ActionDescriptor> outgoingActions = _workflowStepDAO.getOutgoingActions(stepId, workflowDescriptor);
        for (ActionDescriptor action : outgoingActions)
        {
            String stepLabel = _i18nHelper.translateKey(workflowName, _workflowStepDAO.getStepLabel(workflowDescriptor, stepId), WorkflowStepDAO.DEFAULT_STEP_NAME);
            stepLabel = _getStringWithEscapedSpace(stepLabel);
            String actionLabel = _workflowTransitionDAO.getActionLabel(workflowName, action);
            actionLabel = _getStringWithEscapedSpace(actionLabel);
            String actionNode = _getMindMapNodeContent(
                "++",
                _getActionLabel(workflowName, action),
                _workflowTransitionDAO.getActionIconPathAsBase64(workflowName, action),
                null,
                _getJsFunction(workflowName, String.valueOf(stepId), stepLabel, String.valueOf(action.getId()), actionLabel),
                _i18nUtils.translate(new I18nizableText("plugin.workflow", "PLUGIN_WORKFLOW_LINK_SEE_TRANSITION")),
                true
            );
            content.append(actionNode);
        }
        return content.toString();
    }
}
