/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.store;

import javax.jcr.Node;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.version.VersionAwareAmetysObject;
import org.ametys.plugins.workflow.repository.WorkflowAwareAmetysObject;

import com.opensymphony.workflow.StoreException;
import com.opensymphony.workflow.spi.WorkflowEntry;

/**
 * Workflow store that creates local entries for an Ametys object.
 * Each entry belong to an ametys object as soon as the entry is created.
 */
public class AmetysObjectWorkflowStore extends AbstractJackrabbitWorkflowStore
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The ametys object */
    protected WorkflowAwareAmetysObject _ametysObject;
    
    /** Temporary entry created in memory */
    protected WorkflowEntry _inMemoryEntry;
    
    private boolean _preserveHistory;
    
    /**
     * Creates a workflow store for Ametys object. 
     * The history steps will be clear on workflow completion.
     * @param repository the JCR Repository to use.
     * @param ametysObject The ametys object for this store. Can be null in case of an object creation.
     * @param resolver the ametys object resolver
     */
    public AmetysObjectWorkflowStore(Repository repository, WorkflowAwareAmetysObject ametysObject, AmetysObjectResolver resolver)
    {
        this(repository, ametysObject, resolver, false);
    }
    
    /**
     * Creates a workflow store for Ametys object.
     * @param repository the JCR Repository to use.
     * @param ametysObject The ametys object for this store. Can be null in case of an object creation.
     * @param resolver the ametys object resolver
     * @param preserveHistory Set to true to preserve history steps when workflow is complete.
     */
    public AmetysObjectWorkflowStore(Repository repository, WorkflowAwareAmetysObject ametysObject, AmetysObjectResolver resolver, boolean preserveHistory)
    {
        super(repository);
        _resolver = resolver;
        _ametysObject = _ensureBaseVersion(ametysObject);
        _preserveHistory = preserveHistory;
    }
    
    @Override
    public boolean shouldClearHistory()
    {
        return !_preserveHistory;
    }
    
    /**
     * Ametys object getter
     * @return The ametys object
     */
    public WorkflowAwareAmetysObject getAmetysObject()
    {
        return _ametysObject;
    }
    
    @Override
    protected Session _getSession() throws RepositoryException
    {
        return _ametysObject != null ? _ametysObject.getNode().getSession() : null;
    }
    
    @Override
    protected void _release(Session session)
    {
        // do nothing since the store use the ametys object session
    }
    
    /**
     * Provides the ametys object node in the current session
     * @return node
     * @throws RepositoryException on repository error
     */
    protected Node _getAmetysObjectNode() throws RepositoryException
    {
        return _ametysObject.getNode();
    }
    
    @Override
    protected void _createRootNode() throws RepositoryException
    {
        if (_ametysObject == null)
        {
            if (_log.isDebugEnabled())
            {
                _log.debug("Ametys object does not currently exist. Root node creation will be delayed. The creation will be done once the ametys object is available.");
            }
            
            return;
        }
        
        Session session = null;
        try
        {
            session = _getSession();
                
            Node contentNode = _getAmetysObjectNode();
            
            if (!contentNode.hasNode("ametys-internal:workflows"))
            {
                if (_log.isDebugEnabled())
                {
                    _log.debug(String.format("Creating osworkflow root node for the ametys object '%s'.", _ametysObject.getId()));
                }
                
                contentNode.addNode("ametys-internal:workflows");
            }
            else
            {
                if (_log.isDebugEnabled())
                {
                    _log.debug(String.format("Existing osworkflow root node of the ametys object '%s', skipping creation.", _ametysObject.getId()));
                }
            }
        }
        finally
        {
            _release(session);
        }
    }
    
    @Override
    protected Node _getRootNode(Session session) throws RepositoryException
    {
        Node contentNode = _getAmetysObjectNode();
        return contentNode.getNode("ametys-internal:workflows");
    }
    
    @Override
    protected Node _getOrCreateParentEntryNode(Node root, long id) throws RepositoryException
    {
        return root;
    }
    
    @Override
    protected synchronized long _getNextEntryId() throws RepositoryException
    {
        if (_ametysObject == null)
        {
            // default workflow entry id
            return 0;
        }
        
        return super._getNextEntryId();
    }
    
    /**
     * Store a new workflow entry into the JCR repository
     * @param workflowEntry The new entry
     * @throws StoreException on error
     */
    @Override
    protected void storeNewEntry(WorkflowEntry workflowEntry) throws StoreException
    {
        // Temporary store in memory if ametys object does not exist currently.
        // Otherwise persist the entry into the JCR repository
        if (_ametysObject == null)
        {
            _inMemoryEntry = workflowEntry;
            return;
        }
        
        super.storeNewEntry(workflowEntry);
    }
    
    /**
     * Bind an ametys object to this store. Must be done in case of an object
     * creation. As soon as the ametys object is created, the store must be
     * notified to persist the workflow entry into the repository.
     * @param ametysObject The ametys object to bind
     * @throws StoreException on error
     */
    public void bindAmetysObject(WorkflowAwareAmetysObject ametysObject) throws StoreException
    {
        _ametysObject = _ensureBaseVersion(ametysObject);
        
        if (_inMemoryEntry != null)
        {
            // Create root node
            try
            {
                _createRootNode();
            }
            catch (RepositoryException e)
            {
                throw new StoreException("Unable to create the root node", e);
            }
            
            // Persist the memory entry into the repository
            storeNewEntry(_inMemoryEntry);
            _inMemoryEntry = null;
        }
    }
    
    private WorkflowAwareAmetysObject _ensureBaseVersion(WorkflowAwareAmetysObject ametysObject)
    {
        if (ametysObject instanceof VersionAwareAmetysObject vao && vao.getRevision() != null)
        {
            // Get another instance of the ametys object to switch its revision and not the current object one's
            WorkflowAwareAmetysObject workflowAndVersionAwareAmetysObject = _resolver.resolveById(ametysObject.getId());
            return workflowAndVersionAwareAmetysObject;
        }
        else
        {
            return ametysObject;
        }
    }
    
}
