/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.support;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.plugins.workflow.EnhancedConditionExtensionPoint;
import org.ametys.plugins.workflow.EnhancedFunctionExtensionPoint;

import com.opensymphony.workflow.Condition;
import com.opensymphony.workflow.FunctionProvider;
import com.opensymphony.workflow.Register;
import com.opensymphony.workflow.TypeResolver;
import com.opensymphony.workflow.Validator;
import com.opensymphony.workflow.WorkflowException;

/**
 * Avalon-aware type resolver.
 * <p>
 * Used for retreiving avalon component using the {@link ServiceManager} and a
 * role.
 */
public class AvalonTypeResolver extends TypeResolver
{
    private static final String __AVALON_TYPE = "avalon";
    private static final String __ROLE = "role";
    private static final String __ID = "id";
    private ServiceManager _serviceManager;

    private EnhancedFunctionExtensionPoint _enhancedFunctionExtensionPoint;
    private EnhancedConditionExtensionPoint _enhancedConditionExtensionPoint;
    /**
     * Create an avalon type resolver.
     * 
     * @param serviceManager the service manager.
     */
    public AvalonTypeResolver(ServiceManager serviceManager)
    {
        _serviceManager = serviceManager;
        try
        {
            _enhancedFunctionExtensionPoint = (EnhancedFunctionExtensionPoint) serviceManager.lookup(EnhancedFunctionExtensionPoint.ROLE);
            _enhancedConditionExtensionPoint = (EnhancedConditionExtensionPoint) serviceManager.lookup(EnhancedConditionExtensionPoint.ROLE);
        }
        catch (ServiceException e)
        {
            throw new RuntimeException("Unable to lookup EnhancedFunctionExtensionPoint", e);
        }
    }

    @Override
    public Condition getCondition(String type, Map args) throws WorkflowException
    {
        if (__AVALON_TYPE.equals(type))
        {
            return (Condition) _getConditionExtension((String) args.get(__ID));
        }

        return super.getCondition(type, args);
    }

    @Override
    public FunctionProvider getFunction(String type, Map args) throws WorkflowException
    {
        if (__AVALON_TYPE.equals(type))
        {
            return (FunctionProvider) _getFunctionExtension((String) args.get(__ID));
        }

        return super.getFunction(type, args);
    }

    @Override
    public Register getRegister(String type, Map args) throws WorkflowException
    {
        if (__AVALON_TYPE.equals(type))
        {
            return (Register) _getComponent((String) args.get(__ROLE));
        }

        return super.getRegister(type, args);
    }

    @Override
    public Validator getValidator(String type, Map args) throws WorkflowException
    {
        if (__AVALON_TYPE.equals(type))
        {
            return (Validator) _getComponent((String) args.get(__ROLE));
        }

        return super.getValidator(type, args);
    }

    private Object _getComponent(String role) throws WorkflowException
    {
        if (role == null)
        {
            throw new WorkflowException("Missing 'role' argument for avalon type");
        }
        
        try
        {
            return _serviceManager.lookup(role);
        }
        catch (ServiceException e)
        {
            throw new WorkflowException("Unable to retrieve avalon component for role: '" + role + "'", e);
        }
    }
    
    private Object _getFunctionExtension(String id) throws WorkflowException
    {
        if (id == null)
        {
            throw new WorkflowException("Missing 'id' argument for avalon type");
        }
        
        return _enhancedFunctionExtensionPoint.getExtension(id);
    }
    
    private Object _getConditionExtension(String id) throws WorkflowException
    {
        if (id == null)
        {
            throw new WorkflowException("Missing 'id' argument for avalon type");
        }
        
        return _enhancedConditionExtensionPoint.getExtension(id);
    }
}
