/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.support;

import java.util.Map;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * This class is intended to be use as a simple helper to construct Map for i18n
 * message from SAX events. <br>
 * Each pair key/value will be put (as Strings) in the constructed Map
 */
public class I18nMessageHandler extends DefaultHandler
{
    /** The object being constructed */
    private Map<I18nizableText, String> _map;

    /** Current characters from SAX events */
    private StringBuilder _currentText;

    /** The catalog of the messages */
    private String _catalog;
    
    /** The current key message */
    private String _currentKey;

    /** True when saxing element with "message" qName */
    private boolean _message;

    /**
     * Construct a I18nMessageHandler
     * 
     * @param i18nMessages The map representing each pair of key/value for i18n message
     * @param catalog The catalog of the messages
     */
    public I18nMessageHandler(Map<I18nizableText, String> i18nMessages, String catalog)
    {
        this._map = i18nMessages;
        this._catalog = catalog;
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
    {
        _message = false;
        if (qName.equals("message"))
        {
            _message = true;
            _currentText = new StringBuilder();
            _currentKey = attributes.getValue("key");
        }
    }

    @Override
    public void characters(char[] ch, int start, int length) throws SAXException
    {
        if (_message)
        {
            _currentText.append(ch, start, length);
        }
    }

    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException
    {
        if (qName.equals("message"))
        {
            _map.put(new I18nizableText(_catalog, _currentKey), _currentText.toString());
            _message = false;
        }
    }
}
