/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The workflow condition DAO
 */
Ext.define('Ametys.plugins.workflow.dao.WorkflowResultDAO', {
    singleton: true,
    
    constructor: function()
    {
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowResultDAO
         * @method getConditionalResultsToJson 
         * Get a list of available step
         * This calls the method 'getConditionalResultsToJson' of the server DAO 'WorkflowResultDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowResultDAO",
                methodName: "getConditionalResultsToJson",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_RESULT_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowResultDAO
         * @method addConditionalResult 
         * Add workflow conditional result to action
         * This calls the method 'addOperator' of the server DAO 'WorkflowResultDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowResultDAO",
                methodName: "addConditionalResult",
                callback: {
                    handler: this._addConditionalResultCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowResultDAO
         * @method editConditionalResult 
         * Edit a workflow conditional result
         * This calls the method 'editConditionalResult' of the server DAO 'WorkflowResultDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowResultDAO",
                methodName: "editConditionalResult",
                callback: {
                    handler: this._editConditionalResultCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowResultDAO
         * @method deleteConditionalResult 
         * Delete a workflow conditional result from selected action
         * This calls the method 'deleteConditionalResult' of the server DAO 'WorkflowResultDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowResultDAO",
                methodName: "deleteConditionalResult",
                callback: {
                    handler: this._deleteConditionalResultCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
    },
    
    /**
     * Callback function called after {@link #deleteConditionalResult} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _deleteConditionalResultCB: function(response)
    {
        var resultTarget = this._getResultTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.DELETED,
            targets: [resultTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * Callback function called after {@link #editConditionalResult} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _editConditionalResultCB: function(response)
    {
        var resultTarget = this._getResultTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: [resultTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * Callback function called after {@link #addConditionalResult} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _addConditionalResultCB: function(response)
    {
        if (response.message == 'duplicate-state')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_ADD_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_FINAL_STATES_DUPLICATE_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {   
            var resultTarget = this._getResultTarget(response)
           
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.CREATED,
                targets: [resultTarget]
            });
            
            this._sendWorkflowModifiedMessage(response);
        }
    },
    
    /**
     * @private
     * Get the result target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.actionId the id of the action
     * @param {String} response.nodeId the id of result node in the result tree
     */
    _getResultTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_RESULT,
            parameters: {
                isConditional: response.isConditional,
                nodeLabel: response.nodeLabel,
                nodeId: response.nodeId,
                actionId: response.actionId,
                stepId: response.stepId,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Get the action target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.stepId the id of the parent step
     * @param {String} response.actionId the id of the action
     */
    _getActionTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_ACTION,
            parameters: {
                id: response.actionId,
                label:response.actionLabel,
                stepId: response.stepId,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Get the step target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.stepId the id of the step
     */
    _getStepTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_STEP,
            parameters: {
                id: response.stepId,
                label:response.stepLabel,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Send a message modified for workflow target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     */
    _sendWorkflowModifiedMessage: function(response)
    {
        var workflowTarget = {
            id: Ametys.message.MessageTarget.WORKFLOW_OBJECT,
            parameters: {
                id: response.workflowId
            }
        };
        var stepTarget = this._getStepTarget(response);
        stepTarget.subtargets = this._getActionTarget(response);
        workflowTarget.subtargets = stepTarget;
            
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: [workflowTarget]
        });
    }
});
