/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow;

import com.opensymphony.workflow.query.Expression;

/**
 * PropertySet expressions are used when constructing a workflow query
 * on a property associated with a workflow.<br>
 * PropertySet expressions have three attributes.<br>
 * These are:
 * <ul>
 * <li>operator: This is the operator to apply on the expression.
 * <li>type: The PropertySet item type to test agains.
 * <li>key: The PropertySet item key to test agains.
 * </ul>
 * For the moment, only string type is supported.<br>
 * <b>Warning: PropertySet expressions can only be nested with others PropertySet
 * expressions (not with FieldExpression)!</b>
 */
public class PropertySetExpression extends Expression
{
    /**
     * Constant for the equality operator.
     */
    public static final int EQUALS = 1;

    /**
     * Constant for the wildcard equality operator.
     */
    public static final int WILDCARD_EQUALS = 2;

    private Object _value;

    private String _key;

    private int _type;

    private int _operator;

    /**
     * Create a property set expression.
     * @param operator The operator.
     * @param type The type of the operand.
     * @param key The item key to test agains.
     * @param value The operand value.
     */
    public PropertySetExpression(int operator, int type, String key, Object value)
    {
        _operator = operator;
        _type = type;
        _key = key;
        _value = value;
    }

    /**
     * Create a negative property set expression.
     * @param operator The operator.
     * @param type The type of the operand.
     * @param key The item key to test agains.
     * @param value The operand value.
     * @param isNegated The negative state of the operator.
     */
    public PropertySetExpression(int operator, int type, String key, Object value, boolean isNegated)
    {
        this(operator, type, key, value);
        super.negate = isNegated;
    }

    /**
     * Get the operator of the expression.
     * @return The operator.
     */
    public int getOperator()
    {
        return _operator;
    }

    /**
     * Set the operator of the expression.
     * @param operator The operator.
     */
    public void setOperator(int operator)
    {
        _operator = operator;
    }

    /**
     * Get the type of the operand.
     * @return The type.
     */
    public int getType()
    {
        return _type;
    }

    /**
     * Set the type of the operand.
     * @param type The type.
     */
    public void setType(int type)
    {
        _type = type;
    }

    /**
     * Get the operator of the expression.
     * @return The operator.
     */
    public String getKey()
    {
        return _key;
    }

    /**
     * Set the key to test agains.
     * @param key The key.
     */
    public void setKey(String key)
    {
        _key = key;
    }

    /**
     * Get the value to test agains.
     * @return The value.
     */
    public Object getValue()
    {
        return _value;
    }

    /**
     * Set the value to test agains.
     * @param value The value.
     */
    public void setValue(Object value)
    {
        _value = value;
    }

    @Override
    public boolean isNested()
    {
        return false;
    }
}
