/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.cocoon;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.AbstractWorkflowComponent.ConditionFailure;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.parameter.ValidationResult;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.InvalidInputException;
import com.opensymphony.workflow.WorkflowException;

/**
 * Abstract action for managing a workflow instance.
 * The following parameters are supported:
 * <dl>
 *  <dt>actionId
 *  <dd>the id of the action to fire
 * </dl>
 */
public abstract class AbstractWorkflowAction extends ServiceableAction implements Initializable
{
    /** Workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    @Override
    public void initialize() throws Exception
    {
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        int actionId = _getActionId(objectModel, source, parameters);
        Map inputs = _getInputs(redirector, objectModel, source, parameters);
        
        try
        {
            Map result = _act(redirector, objectModel, source, parameters, actionId, inputs);
            
            @SuppressWarnings("unchecked")
            Map<String, Object> resultMap = (Map<String, Object>) inputs.get(AbstractWorkflowComponent.RESULT_MAP_KEY);
            resultMap.put("success", true);
            
            return result;
        }
        catch (InvalidActionException e)
        {
            return _processInvalidActionException(redirector, objectModel, source, parameters, actionId, inputs, e);
        }
        catch (WorkflowException e)
        {
            return _processWorkflowException(redirector, objectModel, source, parameters, actionId, e);
        }
    }

    /**
     * Provide the action id to use.
     * Default implementation uses parameter <code>actionId</code>.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @return the action id.
     * @throws Exception if an error occurs.
     */
    protected int _getActionId(Map objectModel, String source, Parameters parameters) throws Exception
    {
        return parameters.getParameterAsInteger("actionId", Integer.MIN_VALUE);
    }

    /**
     * Provide the inputs to use.
     * Default implementation provide the redirector.
     * @param redirector the redirector.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @return the inputs to use.
     * @throws Exception if an error occurs.
     */
    protected Map<String, Object> _getInputs(Redirector redirector, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> inputs = new HashMap<>();        
        inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, objectModel.get(ObjectModelHelper.PARENT_CONTEXT));
        inputs.put(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY, new ArrayList<>());
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        if (request.getAttribute(AuthenticateAction.REQUEST_ATTRIBUTE_INTERNAL_ALLOWED) != null)
        {
            inputs.put(CheckRightsCondition.FORCE, true);
        }
        
        // Provide a map for providing data to the generator
        Map<String, Object> result = new LinkedHashMap<>();
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        inputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, result);
        
        return inputs;
    }

    /**
     * Initialize or act on a workflow instance.
     * @param redirector the redirector.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @param actionId the action id to use.
     * @param inputs the inputs to use
     * @return the action result.
     * @throws InvalidInputException if the action id is not valid.
     * @throws WorkflowException if the action failed.
     */
    protected abstract Map _act(Redirector redirector, Map objectModel, String source, Parameters parameters, int actionId, Map inputs) throws InvalidInputException, WorkflowException;

    /**
     * Called when the current action is invalid.
     * Default implementation throw an exception.
     * @param redirector the redirector.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @param actionId the invalid action id.
     * @param inputs the inputs
     * @param e the invalid action exception.
     * @return the action result.
     * @throws Exception in order to stop the current pipeline.
     */
    protected Map _processInvalidActionException(Redirector redirector, Map objectModel, String source, Parameters parameters, long actionId, Map inputs, InvalidActionException e) throws Exception
    {
        if (_getWorkflowErrors(inputs) != null)
        {
            // Errors was traited in result map
            getLogger().error(String.format("Invalid action: '%d' for %s. %s", actionId, _getExceptionContext(objectModel, source, parameters), _getFailConditions(inputs)), e);
            return Collections.EMPTY_MAP;
        }
        else
        {
            // The error was not traited
            throw new Exception(String.format("Invalid action: '%d' for %s. %s", actionId, _getExceptionContext(objectModel, source, parameters), _getFailConditions(inputs)), e);
        }
    }
    
    /**
     * Get the fail conditions as a String
     * @param inputs the inputs
     * @return the fail conditions
     */
    protected String _getFailConditions (Map inputs)
    {
        if (inputs.containsKey(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY))
        {
            @SuppressWarnings("unchecked")
            List<ConditionFailure> failConditions = (List<ConditionFailure>) inputs.get(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY);
            if (failConditions.size() > 0)
            {
                return String.format("Fail conditions are: %s", StringUtils.join(failConditions.stream().map(ConditionFailure::text).toList(), " / "));
            }
        }
        
        return "";
    }
    
    /**
     * Get the workflow errors
     * @param inputs The inputs
     * @return The workflow errors or null if not found
     */
    protected List<I18nizableText> _getWorkflowErrors (Map inputs)
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> result = (Map<String, Object>) inputs.get(AbstractWorkflowComponent.RESULT_MAP_KEY);
        
        if (result.containsKey(AbstractWorkflowComponent.WORKFLOW_VALIDATION_KEY))
        {
            return ((ValidationResult) result.get(AbstractWorkflowComponent.WORKFLOW_VALIDATION_KEY)).getErrors();
        }
        
        return null;
    }
    
    /**
     * Get the workflow warnings
     * @param inputs The inputs
     * @return The workflow warnings or null if not found
     */
    @SuppressWarnings("unchecked")
    protected List<I18nizableText> _getWorkflowWarns (Map inputs)
    {
        Map<String, Object> result = (Map<String, Object>) inputs.get(AbstractWorkflowComponent.RESULT_MAP_KEY);
        
        if (result.containsKey(AbstractWorkflowComponent.WORKFLOW_VALIDATION_KEY))
        {
            return ((ValidationResult) result.get(AbstractWorkflowComponent.WORKFLOW_VALIDATION_KEY)).getWarnings();
        }
        
        return null;
    }

    /**
     * Called when the current action has thrown a {@link WorkflowException}.
     * Default implementation throw an exception.
     * @param redirector the redirector.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @param actionId the invalid action id.
     * @param e the workflow exception.
     * @return the action result.
     * @throws Exception in order to stop the current pipeline.
     */
    protected Map _processWorkflowException(Redirector redirector, Map objectModel, String source, Parameters parameters, long actionId, WorkflowException e) throws Exception
    {
        throw new Exception(String.format("Unable to perform action: '%d' for %s", actionId, _getExceptionContext(objectModel, source, parameters)), e);
    }

    /**
     * Provides the exception context for a more friendly error message.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @return the message describing the exception context.
     */
    protected abstract String _getExceptionContext(Map objectModel, String source, Parameters parameters);
}
