/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workflow.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.workflow.EnhancedCondition;
import org.ametys.plugins.workflow.EnhancedConditionExtensionPoint;
import org.ametys.plugins.workflow.ModelItemTypeExtensionPoint;
import org.ametys.plugins.workflow.component.WorkflowArgument;
import org.ametys.plugins.workflow.support.AvalonTypeResolver;
import org.ametys.plugins.workflow.support.WorflowRightHelper;
import org.ametys.plugins.workflow.support.WorkflowElementDefinitionHelper;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowHelper.WorkflowVisibility;
import org.ametys.plugins.workflow.support.WorkflowSessionHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.disableconditions.VoidRelativeDisableCondition;
import org.ametys.runtime.model.disableconditions.VoidRelativeDisableConditions;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.Condition;
import com.opensymphony.workflow.TypeResolver;
import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.loader.AbstractDescriptor;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.ConditionDescriptor;
import com.opensymphony.workflow.loader.ConditionalResultDescriptor;
import com.opensymphony.workflow.loader.ConditionsDescriptor;
import com.opensymphony.workflow.loader.DescriptorFactory;
import com.opensymphony.workflow.loader.RestrictionDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * DAO for workflow conditions
 */
public class WorkflowConditionDAO extends AbstractLogEnabled implements Component, Serviceable
{
    /** The component role */ 
    public static final String ROLE = WorkflowConditionDAO.class.getName();
    
    /** The "and" type of condition */
    public static final String AND = "AND";
    
    /** The "or" type of condition */
    public static final String OR = "OR";
    
    /** Key for "and" label in tree  */
    protected static final String __ANDI18N = "PLUGIN_WORKFLOW_TRANSITION_CONDITIONS_TYPE_AND";
    
    /** Key for "or" label in tree  */
    protected static final String __ORI18N = "PLUGIN_WORKFLOW_TRANSITION_CONDITIONS_TYPE_OR";
    
    /** Extension point for workflow arguments data type */
    protected static ModelItemTypeExtensionPoint _workflowArgumentDataTypeExtensionPoint;
    
    private static final String __OR_ROOT_OPERATOR = "or0";
    private static final String __AND_ROOT_OPERATOR = "and0";
    private static final String __STEP_RESULT_PREFIX = "step";
    private static final String __ROOT_RESULT_ID = "root";
    private static final String __ATTRIBUTE_CONDITIONS_LIST = "conditions-list";
    
    /** The workflow session helper */
    protected WorkflowSessionHelper _workflowSessionHelper;
    
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    /** The workflow right helper */
    protected WorflowRightHelper _workflowRightHelper;
    
    /** The workflow result helper */
    protected WorkflowResultDAO _workflowResultDAO;
    
    /** The workflow step DAO */
    protected WorkflowStepDAO _workflowStepDAO;
    
    /** The workflow transition DAO */
    protected WorkflowTransitionDAO _workflowTransitionDAO;
    
    /** Extension point for Conditions */
    protected EnhancedConditionExtensionPoint _enhancedConditionExtensionPoint;
    
    /** The service manager */
    protected ServiceManager _manager;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _workflowHelper = (WorkflowHelper) smanager.lookup(WorkflowHelper.ROLE);
        _workflowRightHelper = (WorflowRightHelper) smanager.lookup(WorflowRightHelper.ROLE);
        _workflowSessionHelper = (WorkflowSessionHelper) smanager.lookup(WorkflowSessionHelper.ROLE);
        _workflowResultDAO = (WorkflowResultDAO) smanager.lookup(WorkflowResultDAO.ROLE);
        _workflowStepDAO = (WorkflowStepDAO) smanager.lookup(WorkflowStepDAO.ROLE);
        _workflowTransitionDAO = (WorkflowTransitionDAO) smanager.lookup(WorkflowTransitionDAO.ROLE);
        _workflowArgumentDataTypeExtensionPoint = (ModelItemTypeExtensionPoint) smanager.lookup(ModelItemTypeExtensionPoint.ROLE_WORKFLOW);
        _enhancedConditionExtensionPoint = (EnhancedConditionExtensionPoint) smanager.lookup(EnhancedConditionExtensionPoint.ROLE);
        _manager = smanager;
    }
    
    /**
     * Get the condition's model items as fields to configure edition form panel
     * @return the parameters field as Json readable map
     * @throws ProcessingException exception while saxing view to json
     */
    @Callable(rights = {"Workflow_Right_Edit", "Workflow_Right_Edit_User"})
    public Map<String, Object> getConditionsModel() throws ProcessingException
    {
        Map<String, Object> response = new HashMap<>();
        
        View view = new View();
        SimpleViewItemGroup fieldset = new SimpleViewItemGroup();
        fieldset.setName("conditions");
        
        Set<Pair<String, EnhancedCondition>> conditions = _enhancedConditionExtensionPoint.getAllConditions()
                .stream()
                .filter(this::_hasConditionRight)
                .collect(Collectors.toSet());
        
        List<ElementDefinition> modelItems = new ArrayList<>();
        modelItems.add(_getConditionsListModelItem(conditions));
        modelItems.addAll(_getArgumentModelItems(conditions));
        Model.of("conditions", "worklow", modelItems.toArray(new ElementDefinition[modelItems.size()]));
        
        for (ElementDefinition conditionArgument : modelItems)
        {
            ViewElement argumentView = new ViewElement();
            argumentView.setDefinition(conditionArgument);
            fieldset.addViewItem(argumentView);
        }
        
        view.addViewItem(fieldset);

        response.put("parameters", view.toJSON(DefinitionContext.newInstance().withEdition(true)));
        
        return response;
    }
    
    private boolean _hasConditionRight(Pair<String, EnhancedCondition> condition)
    {
        List<WorkflowVisibility> conditionVisibilities = condition.getRight().getVisibilities();
        if (conditionVisibilities.contains(WorkflowVisibility.USER))
        {
            return _workflowRightHelper.hasEditUserRight();
        }
        else if (conditionVisibilities.contains(WorkflowVisibility.SYSTEM))
        {
            return _workflowRightHelper.hasEditSystemRight();
        }
        return false;
    }

    
    /**
     * Get the model item for the list of conditions
     * @param conditions a list of all the conditions with their ids
     * @return an enum of the conditions as a model item 
     */
    private ElementDefinition<String> _getConditionsListModelItem(Set<Pair<String, EnhancedCondition>> conditions)
    {
        ElementDefinition<String> conditionsList = WorkflowElementDefinitionHelper.getElementDefinition(
            __ATTRIBUTE_CONDITIONS_LIST,
            new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_ADD_CONDITION_DIALOG_ROLE"),
            new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_ADD_CONDITION_DIALOG_ROLE_DESC"),
            true,
            false
        );
        
        StaticEnumerator<String> conditionStaticEnumerator = new StaticEnumerator<>();
        for (Pair<String, EnhancedCondition> condition : conditions)
        {
            conditionStaticEnumerator.add(condition.getRight().getLabel(), condition.getLeft());
        }
        conditionsList.setEnumerator(conditionStaticEnumerator);
       
        return conditionsList;
    }

    /**
     * Get a list of workflow arguments model items with disable conditions on non related function selected
     * @param conditions a list of all the conditions with their ids
     * @return the list of model items
     */
    protected List<WorkflowArgument> _getArgumentModelItems(Set<Pair<String, EnhancedCondition>> conditions)
    {
        List<WorkflowArgument> argumentModelItems = new ArrayList<>();
        for (Pair<String, EnhancedCondition> condition : conditions)
        {
            String conditionId = condition.getLeft();
            DisableConditions disableConditions = new VoidRelativeDisableConditions();
            DisableCondition disableCondition = new VoidRelativeDisableCondition(__ATTRIBUTE_CONDITIONS_LIST, OPERATOR.NEQ, conditionId); 
            disableConditions.getConditions().add(disableCondition);
            
            for (WorkflowArgument arg : condition.getRight().getArguments())
            {
                arg.setName(conditionId + "-" + arg.getName());
                arg.setDisableConditions(disableConditions);
                argumentModelItems.add(arg);
            }
        }
        return argumentModelItems;
    }
    
    /**
     * Get current argument's values for condition
     * @param workflowName unique name of current workflow
     * @param actionId id of current action
     * @param conditionNodeId id of selected node
     * @return a map of the condition's arguments and their current values
     * @throws ProcessingException exception while resolving condition
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getConditionParametersValues(String workflowName, Integer actionId, String conditionNodeId) throws ProcessingException
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        _workflowRightHelper.checkEditRight(workflowDescriptor);

        ConditionDescriptor condition = _getCondition(workflowDescriptor, actionId, conditionNodeId);
        Map<String, String> args = new HashMap<>();
        Map<String, String> conditionCurrentArguments = condition.getArgs();
        String conditionId = conditionCurrentArguments.get("id");
        for (Entry<String, String> entry : conditionCurrentArguments.entrySet())
        {
            String argName = entry.getKey().equals("id") ? __ATTRIBUTE_CONDITIONS_LIST : conditionId + "-" + entry.getKey();
            args.put(argName, entry.getValue());
        }
        args.putAll(conditionCurrentArguments);

        Map<String, Object> results = new HashMap<>();
        results.put("parametersValues", args);
        return results;
    }

    /**
     * Get the current condition as condition descriptor
     * @param workflowDescriptor current workflow
     * @param actionId id of current action
     * @param conditionNodeId id of current node
     * @return the conditions
     */
    protected ConditionDescriptor _getCondition(WorkflowDescriptor workflowDescriptor, Integer actionId, String conditionNodeId)
    {
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        
        String[] path = _workflowResultDAO.getPath(conditionNodeId);
        int nodeToEdit = Integer.valueOf(path[path.length - 1].substring("condition".length()));
        boolean isResult = path[0].startsWith(__STEP_RESULT_PREFIX);
        
        ConditionsDescriptor rootOperator = isResult 
                ? (ConditionsDescriptor) _workflowResultDAO.getRootResultConditions(action.getConditionalResults(), _getStepId(conditionNodeId)).get(0)
                : action.getRestriction().getConditionsDescriptor();
        String parentNodeId = conditionNodeId.substring(0, conditionNodeId.lastIndexOf('-'));
        ConditionsDescriptor parentNode = _getConditionsNode(rootOperator, parentNodeId);
        ConditionDescriptor condition = (ConditionDescriptor) parentNode.getConditions().get(nodeToEdit);
        
        return condition;
    }
    
    /**
     * Edit the condition
     * @param workflowName unique name of current workflow
     * @param stepId id of step parent
     * @param actionId id of current action
     * @param conditionNodeId id of selected node
     * @param arguments map of arguments name and values
     * @return map of condition's info
     * @throws WorkflowException exception while resolving condition
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editCondition(String workflowName, Integer stepId, Integer actionId, String conditionNodeId, Map<String, Object> arguments) throws WorkflowException
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ConditionDescriptor condition = _getCondition(workflowDescriptor, actionId, conditionNodeId);
        _updateArguments(condition, arguments);
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        return _getConditionProperties(workflowDescriptor, action, condition, stepId, conditionNodeId);
    }
    
    private Map<String, Object> _getConditionProperties(WorkflowDescriptor workflow, ActionDescriptor action, ConditionDescriptor condition, Integer stepId, String conditionNodeId)
    {
        Map<String, Object> results = new HashMap<>();
        results.put("conditionId", condition.getArgs().get("id"));
        results.put("nodeId", conditionNodeId);
        results.put("actionId", action.getId());
        results.put("actionLabel", _workflowTransitionDAO.getActionLabel(action));
        results.put("stepId", stepId);
        results.put("stepLabel", _workflowStepDAO.getStepLabel(workflow, stepId));
        results.put("workflowId", workflow.getName());
        return results;
    }
    
    private void _updateArguments(ConditionDescriptor condition, Map<String, Object> arguments)
    {
        Map<String, String> args = condition.getArgs();
        args.clear();
        args.putAll(_getConditionParamsValuesAsString(arguments));
    }
    
    /**
     * Add a new condition
     * @param workflowName unique name of current workflow
     * @param stepId id of step parent
     * @param actionId id of current action
     * @param parentNodeId id of selected node : can be null if no node is selected
     * @param arguments map of arguments name and values
     * @return map of condition's info
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addCondition(String workflowName, Integer stepId, Integer actionId, String parentNodeId, Map<String, Object> arguments) 
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);

        ActionDescriptor action = workflowDescriptor.getAction(actionId);

        boolean isResultCondition = !StringUtils.isBlank(parentNodeId) && parentNodeId.startsWith(__STEP_RESULT_PREFIX);
        String computedParentNodeId = StringUtils.isBlank(parentNodeId) 
                ? __AND_ROOT_OPERATOR 
                : isResultCondition && parentNodeId.split("-").length == 1
                    ? parentNodeId + "-" + __AND_ROOT_OPERATOR
                    : parentNodeId;
        ConditionsDescriptor rootOperator = isResultCondition
                ? _getResultConditionsRootOperator(workflowName, stepId, actionId, parentNodeId, action)
                : _getConditionsRootOperator(workflowName, stepId, actionId, action, parentNodeId);
       
        ConditionsDescriptor currentParentOperator = _getConditionsNode(rootOperator, computedParentNodeId);
        if (StringUtils.isBlank(currentParentOperator.getType()))
        {
            currentParentOperator.setType(AND);
        }
        
        
        List<AbstractDescriptor> conditions = currentParentOperator.getConditions();
        ConditionDescriptor conditionDescriptor = _createCondition(arguments);
        conditions.add(conditionDescriptor);

        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        String conditionNodeId = computedParentNodeId + "-" + "condition" + (conditions.size() - 1);
        return _getConditionProperties(workflowDescriptor, action, conditionDescriptor, stepId, conditionNodeId);
    }
    
    private ConditionDescriptor _createCondition(Map<String, Object> arguments)
    {
        DescriptorFactory factory = new DescriptorFactory();
        ConditionDescriptor conditionDescriptor = factory.createConditionDescriptor();
        conditionDescriptor.setType("avalon");
        _updateArguments(conditionDescriptor, arguments);
        return conditionDescriptor;
    }
    
    /**
     * Get the list of arguments as String, parse multiple arguments
     * @param params List of condition arguments with values
     * @return the map of arguments formated for condition descriptor
     */
    protected Map<String, String> _getConditionParamsValuesAsString(Map<String, Object> params)
    {
        String conditionId = (String) params.get(__ATTRIBUTE_CONDITIONS_LIST);
        EnhancedCondition enhancedCondition = _enhancedConditionExtensionPoint.getExtension(conditionId);
        
        Map<String, String> conditionParams = new HashMap<>();
        conditionParams.put("id", conditionId);
        List<WorkflowArgument> arguments = enhancedCondition.getArguments();
        if (!arguments.isEmpty())
        {
            for (WorkflowArgument argument : arguments)
            {
                String paramKey = conditionId + "-" + argument.getName();
                String paramValue = argument.isMultiple()
                        ?  _getListAsString(params, paramKey)
                        : (String) params.get(paramKey);
                if (!StringUtils.isBlank(paramValue))
                {
                    conditionParams.put(argument.getName(), paramValue);
                }
            }
        }
        return conditionParams;
    }
    
    @SuppressWarnings("unchecked")
    private String _getListAsString(Map<String, Object> params, String paramKey)
    {
        List<String> values = (List<String>) params.get(paramKey);
        return values == null ? "" : String.join(",", values);
    }
    
    /**
     * Get the root operator for regular conditions
     * @param workflowName name of current workflow
     * @param stepId id of step parent
     * @param actionId id of current action
     * @param action the current action
     * @param parentNodeId id of the parent node
     * @return the root conditions operator
     */
    protected ConditionsDescriptor _getConditionsRootOperator(String workflowName, Integer stepId, Integer actionId, ActionDescriptor action, String parentNodeId)
    {
        RestrictionDescriptor restriction = action.getRestriction();
        if (StringUtils.isBlank(parentNodeId)) //root
        {
            if (restriction == null)
            {
                addOperator(workflowName, stepId, actionId, parentNodeId, AND, false);
                restriction = action.getRestriction();
            }
        }
        
        return restriction.getConditionsDescriptor();
    }
    
    /**
     * Get the root operator for result conditions
     * @param workflowName name of current workflow
     * @param stepId id of step parent
     * @param actionId id of current action
     * @param action the current action
     * @param parentNodeId id of the parent node
     * @return the root conditions operator
     */
    protected ConditionsDescriptor _getResultConditionsRootOperator(String workflowName, Integer stepId, Integer actionId, String parentNodeId, ActionDescriptor action)
    {
        int resultStepId = _getStepId(parentNodeId);
        List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
        List<ConditionsDescriptor> resultConditions = _workflowResultDAO.getRootResultConditions(conditionalResults, resultStepId);
        if (resultConditions.isEmpty())
        {
            addOperator(workflowName, stepId, actionId, parentNodeId, AND, true);
            resultConditions = _workflowResultDAO.getRootResultConditions(conditionalResults, resultStepId);
        }
        
        return resultConditions.get(0);
    }
    
    /**
     * Add an operator
     * @param workflowName unique name of current workflow
     * @param stepId id of step parent
     * @param actionId id of current action
     * @param nodeId id of selected node
     * @param operatorType the workflow conditions' type. Can be AND or OR 
     * @param isResultCondition true if parent is a conditional result
     * @return map of the operator's infos
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addOperator(String workflowName, Integer stepId, Integer actionId, String nodeId, String operatorType, boolean isResultCondition)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        
        DescriptorFactory factory = new DescriptorFactory();
        ConditionsDescriptor newOperator = factory.createConditionsDescriptor();
        newOperator.setType(operatorType);
        boolean isRoot = StringUtils.isBlank(nodeId);
        
        String newNodeId = "";
        if (isResultCondition)
        {
            String[] path = _workflowResultDAO.getPath(nodeId);
            List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
            List<ConditionsDescriptor> resultConditions = _workflowResultDAO.getRootResultConditions(conditionalResults, _getStepId(nodeId));
            if (resultConditions.isEmpty())//root
            {
                resultConditions.add(newOperator);
                newNodeId =  nodeId + "-" + operatorType.toLowerCase() + "0";
            }
            else
            {
                isRoot =  path.length == 1;
                ConditionsDescriptor rootOperator = resultConditions.get(0);
                
                ConditionsDescriptor currentOperator = isRoot ? rootOperator : _getConditionsNode(rootOperator, nodeId);
                List<AbstractDescriptor> conditions = currentOperator.getConditions();
                conditions.add(newOperator);
                String parentNodePrefix = isRoot ? nodeId + "-" + __AND_ROOT_OPERATOR : nodeId;
                newNodeId = parentNodePrefix + "-" + operatorType.toLowerCase() + (conditions.size() - 1);
            }
        }
        else
        {
            if (isRoot && action.getRestriction() ==  null) //root with no condition underneath
            {
                RestrictionDescriptor restriction = new RestrictionDescriptor();
                restriction.setConditionsDescriptor(newOperator);
                action.setRestriction(restriction);
                newNodeId = operatorType.toLowerCase() + "0";
            }
            else
            {
                RestrictionDescriptor restriction = action.getRestriction();
                ConditionsDescriptor rootOperator = restriction.getConditionsDescriptor();

                ConditionsDescriptor currentOperator = isRoot ? rootOperator : _getConditionsNode(rootOperator, nodeId);
                List<AbstractDescriptor> conditions = currentOperator.getConditions();
                conditions.add(newOperator);
                String parentNodePrefix = isRoot ? __AND_ROOT_OPERATOR : nodeId;
                newNodeId = parentNodePrefix + "-" + operatorType.toLowerCase() + (conditions.size() - 1);
            }
        }
        
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getOperatorProperties(workflowDescriptor, action, stepId, operatorType, newNodeId);
    }

    private Map<String, Object> _getOperatorProperties(WorkflowDescriptor workflowDescriptor, ActionDescriptor action, Integer stepId, String operatorType, String newNodeId)
    {
        Map<String, Object> results = new HashMap<>();
        results.put("nodeId", newNodeId);
        results.put("type", operatorType);
        results.put("actionId", action.getId());
        results.put("actionLabel", _workflowTransitionDAO.getActionLabel(action));
        results.put("stepId", stepId);
        results.put("stepLabel", _workflowStepDAO.getStepLabel(workflowDescriptor, stepId));
        results.put("workflowId", workflowDescriptor.getName());
        return results;
    }

    private Integer _getStepId(String nodeId)
    {
        String[] path = _workflowResultDAO.getPath(nodeId);
        return Integer.valueOf(path[0].substring(4));
    }
    
    private ConditionsDescriptor _getConditionsNode(ConditionsDescriptor rootOperator, String nodeId)
    {
        ConditionsDescriptor currentOperator = rootOperator;
        
        List<AbstractDescriptor> conditions = rootOperator.getConditions();
        String[] path = _workflowResultDAO.getPath(nodeId);
        boolean isResultCondition = path[0].startsWith(__STEP_RESULT_PREFIX);
        if (!isResultCondition && path.length > 1 || isResultCondition && path.length > 2) //current node is not root direct child
        {
            // get conditions for current node
            int i = isResultCondition ? 2 : 1;
            do
            {
                int currentOrAndConditionIndex = _getConditionIndex(path, i);
                
                currentOperator = (ConditionsDescriptor) conditions.get(currentOrAndConditionIndex);
                conditions = currentOperator.getConditions();
                i++;
            }
            while (i < path.length);
        }
        return currentOperator;
    }
    
    private int _getConditionIndex(String[] path, int conditionIndex)
    {
        String currentConditionId = path[conditionIndex];
        int prefixSize = currentConditionId.startsWith("and")
                ? "and".length()
                : currentConditionId.startsWith("or")
                    ? "or".length()
                    : "condition".length();
        return Integer.valueOf(currentConditionId.substring(prefixSize));
    }
    
    /**
     * Delete operator from action
     * @param workflowName unique name of current workflow
     * @param stepId id of step parent
     * @param actionId id of current action
     * @param nodeId id of selected node
     * @return map of operator's infos
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteOperator(String workflowName, Integer stepId, Integer actionId, String nodeId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        
        String[] path = _workflowResultDAO.getPath(nodeId);
        boolean isResultCondition = nodeId.startsWith(__STEP_RESULT_PREFIX);
        if (isResultCondition)
        {
            _removeResultConditionOperator(nodeId, action, path);
        }
        else
        {
            _removeConditionOperator(nodeId, action, path);
        }
        
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        String type = path[path.length - 1].startsWith("and") ? AND : OR;
        return _getOperatorProperties(workflowDescriptor, action, stepId, type, nodeId);
    }

    /**
     * Remove a regular condition operator 
     * @param nodeId id of operator to remove
     * @param action current action
     * @param path the path to current node
     */
    protected void _removeConditionOperator(String nodeId, ActionDescriptor action, String[] path)
    {
        if (path.length > 1)
        {
            ConditionsDescriptor rootCondition = action.getRestriction().getConditionsDescriptor();
            _removeCondition(rootCondition, nodeId, path);
        }
        else
        {
            action.setRestriction(null);
        }
    }

    /**
     * Remove a result condition operator 
     * @param nodeId id of operator to remove
     * @param action current action
     * @param path the path to current node
     */
    protected void _removeResultConditionOperator(String nodeId, ActionDescriptor action, String[] path)
    {
        if (path.length > 2)
        {
            ConditionsDescriptor rootCondition = (ConditionsDescriptor) _workflowResultDAO.getRootResultConditions(action.getConditionalResults(), _getStepId(nodeId)).get(0);
            _removeCondition(rootCondition, nodeId, path);
        }
        else
        {
            Optional parentConditionalResult = action.getConditionalResults().stream()
                    .filter(cr -> ((ConditionalResultDescriptor) cr).getStep() == _getStepId(nodeId))
                    .findFirst();
            if (parentConditionalResult.isPresent())
            {
                ((ConditionalResultDescriptor) parentConditionalResult.get()).getConditions().clear();
            }
        }
    }

    /**
     * Delete the condition
     * @param workflowName unique name of current workflow
     * @param stepId id of step parent
     * @param actionId id of current action
     * @param nodeId id of selected node
     * @return map of the condition's infos
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteCondition(String workflowName, Integer stepId, Integer actionId, String nodeId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        
        String[] path = _workflowResultDAO.getPath(nodeId);
        boolean isResultCondition = nodeId.startsWith(__STEP_RESULT_PREFIX);
        ConditionsDescriptor rootCondition = isResultCondition 
                ? (ConditionsDescriptor) _workflowResultDAO.getRootResultConditions(action.getConditionalResults(), _getStepId(nodeId)).get(0)
                : action.getRestriction().getConditionsDescriptor();

        int lastIndexOf = nodeId.lastIndexOf('-');
        String parentNodeId = nodeId.substring(0, lastIndexOf);
        ConditionsDescriptor parentNode = _getConditionsNode(rootCondition, parentNodeId);
        int nodeToDeleteIndex = _getConditionIndex(path, path.length - 1);
        List conditions = parentNode.getConditions();
        ConditionDescriptor conditionToRemove = (ConditionDescriptor) conditions.get(nodeToDeleteIndex);
        conditions.remove(conditionToRemove);
        
        if (conditions.isEmpty() && (path.length == 1 || path.length == 2 && path[0].startsWith(AND))) //if only an operator "and' is left
        {
            action.setRestriction(null);
        }
        
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getConditionProperties(workflowDescriptor, action, conditionToRemove, stepId, nodeId);
    }
    
    /**
     * Remove a condition
     * @param rootCondition the first parent condition operator
     * @param conditionNodeId id of the current condition
     * @param conditionPath the full path to the condition
     * @return the list of conditions after removal
     */
    protected List _removeCondition(ConditionsDescriptor rootCondition, String conditionNodeId, String[] conditionPath)
    {
        int lastIndexOf = conditionNodeId.lastIndexOf('-');
        String parentNodeId = conditionNodeId.substring(0, lastIndexOf);
        ConditionsDescriptor parentNode = _getConditionsNode(rootCondition, parentNodeId);
        int nodeToDeleteIndex = _getConditionIndex(conditionPath, conditionPath.length - 1);
        List conditions = parentNode.getConditions();
        conditions.remove(nodeToDeleteIndex);
        
        return conditions;
    }
    
    /**
     * Get the tree's condition nodes
     * @param currentNode id of the current node 
     * @param workflowName unique name of current workflow 
     * @param actionId id of current action
     * @return a map of current node's children
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getConditionNodes(String currentNode, String workflowName, Integer actionId) 
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        List<Map<String, Object>> nodes = new ArrayList<>();
        if (_workflowRightHelper.canRead(workflowDescriptor))
        {
            ActionDescriptor action = workflowDescriptor.getAction(actionId);
            List<AbstractDescriptor> conditions = _getConditions(currentNode, action);
            if (!conditions.isEmpty())
            {
                boolean rootIsAND = !action.getRestriction().getConditionsDescriptor().getType().equals(OR);
                for (int i = 0; i < conditions.size(); i++)
                {
                    nodes.add(conditionToJSON(conditions.get(i), currentNode, i, rootIsAND));
                }
            }
        }
        
        return Map.of("conditions", nodes);
    }
    
    /**
     * Get conditions below current node
     * @param currentNode id of the current node
     * @param action current action
     * @return a list of childnodes condition
     */
    protected List<AbstractDescriptor> _getConditions(String currentNode, ActionDescriptor action)
    {
        RestrictionDescriptor restriction = action.getRestriction();
        if (restriction != null)
        {
            ConditionsDescriptor rootConditionsDescriptor = restriction.getConditionsDescriptor();

            String[] path = _workflowResultDAO.getPath(currentNode);
            // The current node is root and it's a OR node, so display it ...
            if ("root".equals(currentNode) && rootConditionsDescriptor.getType().equals(OR))
            {
                return List.of(rootConditionsDescriptor);
            }
            // ... the current node is a AND node, display child conditions ...
            else if (path.length == 1)
            {
                return rootConditionsDescriptor.getConditions();
            }
            // ... the selected node is not a condition, so it has children
            // we need to search the condition and get child condition of current node
            else if (!path[path.length - 1].startsWith("condition")) 
            {
                List<AbstractDescriptor> conditions = rootConditionsDescriptor.getConditions();
                // get conditions for current node
                int i = 1;
                do
                {
                    String currentOrAndConditionId = path[i];
                    int currentOrAndConditionIndex = (currentOrAndConditionId.startsWith("and"))
                            ? Integer.valueOf(currentOrAndConditionId.substring(3))
                            : Integer.valueOf(currentOrAndConditionId.substring(2));
                    
                    ConditionsDescriptor currentOrAndCondition = (ConditionsDescriptor) conditions.get(currentOrAndConditionIndex);
                    conditions = currentOrAndCondition.getConditions();
                    i++;
                }
                while (i < path.length);

                return conditions;
            }
        }
        
        return List.of();
    }
    
    /**
     * Get condition or condition types properties 
     * @param condition current condition, can be ConditionsDescriptor or ConditionDescriptor 
     * @param currentNodeId the id of the current node in the ConditionTreePanel
     * @param index index of current condition in node's condition list
     * @param rootIsAND true if root node is an "AND" condition (in which case it's hidden and has no id)
     * @return a map of the condition infos
     */
    public Map<String, Object> conditionToJSON(AbstractDescriptor condition, String currentNodeId, int index, boolean rootIsAND) 
    {
        Map<String, Object> infosConditions = new HashMap<>();
        boolean isResult = currentNodeId.startsWith(__STEP_RESULT_PREFIX);
        boolean isRoot = __ROOT_RESULT_ID.equals(currentNodeId) 
                || isResult && _workflowResultDAO.getPath(currentNodeId).length == 1;
        String prefix = isResult && isRoot ? currentNodeId + "-" : "";
        prefix += isRoot && rootIsAND ? __AND_ROOT_OPERATOR : isRoot ? __OR_ROOT_OPERATOR : currentNodeId;
        // if it's a 'and' or a 'or'
        if (condition instanceof ConditionsDescriptor operator)
        {
            String type = ((ConditionsDescriptor) condition).getType();
            if (!type.equals(OR))
            {
                String id = prefix + "-and" + index;
                infosConditions.put("id", id);
                I18nizableText i18nLabel = new I18nizableText("plugin.workflow", __ANDI18N);
                infosConditions.put("label", i18nLabel);
            }
            else
            {
                String id = isRoot && !rootIsAND ? prefix : prefix + "-or" + index;
                infosConditions.put("id", id);
                I18nizableText i18nLabel = new I18nizableText("plugin.workflow", __ORI18N);
                infosConditions.put("label", i18nLabel);
            }
            infosConditions.put("hasChildren", !operator.getConditions().isEmpty());
        }
        else //it's a condition
        {
            String id = prefix + "-condition" + index;
            infosConditions.put("id", id);
            infosConditions.put("conditionId", ((ConditionDescriptor) condition).getArgs().get("id"));
            infosConditions.put("label", _getConditionLabel((ConditionDescriptor) condition));
            infosConditions.put("hasChildren", false);
        }
       
        return infosConditions;
    }
    
    /**
     * Get current condition's label
     * @param workflowName unique name of current workflow
     * @param actionId id of current action
     * @param nodeId id of selected node
     * @return the label
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public I18nizableText getConditionLabel(String workflowName, Integer actionId, String nodeId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkReadRight(workflowDescriptor);
        
        ConditionDescriptor condition = _getCondition(workflowDescriptor, actionId, nodeId);
        return _getConditionLabel(condition);
    }
    
    /**
     * Get if current operator has children conditions
     * @param workflowName unique name of current workflow
     * @param actionId id of current action
     * @param nodeId id of selected operator node
     * @return true if operator has children
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public boolean hasChildConditions(String workflowName, Integer actionId, String nodeId)
    { 
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkReadRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        List<AbstractDescriptor> conditions = nodeId.startsWith("step") 
                ? _workflowResultDAO.getChildrenResultConditions(nodeId, action, action.getConditionalResults())
                : _getConditions(nodeId, action);
        return  !conditions.isEmpty();
    }
    
    /**
     * Get condition's description or id 
     * @param condition the current condition
     * @return the condition description if exist, or its id if not
     */
    protected I18nizableText _getConditionLabel(ConditionDescriptor condition) 
    {
        String id = (String) condition.getArgs().get("id");
        TypeResolver typeResolver = new AvalonTypeResolver(_manager);
        try
        {
            Condition function = typeResolver.getCondition(condition.getType(), condition.getArgs());
            if (function instanceof EnhancedCondition enhancedCondition)
            {
                List<WorkflowArgument> arguments = enhancedCondition.getArguments();
                Map<String, String> values = new HashMap<>();
                for (WorkflowArgument arg : arguments)
                {
                    values.put(arg.getName(), (String) condition.getArgs().get(arg.getName()));
                }
                I18nizableText description = _getDescription(enhancedCondition, values);
                
                return description != null ? description : new I18nizableText(id);
            }
        }
        catch (WorkflowException e)
        {
            getLogger().warn("An error occured while resolving condition with id {}", id, e);
        }
        return new I18nizableText(id);
    }

    private I18nizableText _getDescription(EnhancedCondition function, Map<String, String> values)
    {
        try
        {
            return function.getFullLabel(values);
        }
        catch (Exception e)
        {
            getLogger().warn("Can't get description for condition '{}': a parameter might be missing", function.getClass().getName(), e);
            return null;
        }
    }
}
