/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.workflow.support.WorflowRightHelper;
import org.ametys.plugins.workflow.support.WorkflowSessionHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * DAO for workflow element's meta attributes
 */
public class WorkflowPropertyDAO extends AbstractLogEnabled implements Component, Serviceable
{
    /** The workflow session helper */
    protected WorkflowSessionHelper _workflowSessionHelper;
    
    /** The workflow right helper */
    protected WorflowRightHelper _workflowRightHelper;
    
    /** The workflow step DAO */
    protected WorkflowStepDAO _workflowStepDAO;
    
    /** The workflow transition DAO */
    protected WorkflowTransitionDAO _workflowTransitionDAO;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _workflowSessionHelper = (WorkflowSessionHelper) smanager.lookup(WorkflowSessionHelper.ROLE);
        _workflowRightHelper = (WorflowRightHelper) smanager.lookup(WorflowRightHelper.ROLE);
        _workflowStepDAO = (WorkflowStepDAO) smanager.lookup(WorkflowStepDAO.ROLE);
        _workflowTransitionDAO = (WorkflowTransitionDAO) smanager.lookup(WorkflowTransitionDAO.ROLE);
    }
    
    /**
     * Get properties of current step
     * @param workflowName the workflow unique name
     * @param stepId id of the current step
     * @return a map of the properties
     */
    @SuppressWarnings("unchecked")
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStepProperties(String workflowName, Integer stepId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        List<Map<String, Object>> properties2json = new ArrayList<>();
        if (_workflowRightHelper.canRead(workflowDescriptor))
        {
            StepDescriptor step = workflowDescriptor.getStep(stepId);
            if (step != null)
            {
                properties2json.addAll(_properties2JSON(step.getMetaAttributes()));
            }
        }
        
        return Map.of("data", properties2json);
    }
    
    /**
     * Get properties of current action
     * @param workflowName the workflow unique name
     * @param actionId id of the current action
     * @return a map of the properties
     */
    @SuppressWarnings("unchecked")
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getActionProperties(String workflowName, Integer actionId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        List<Map<String, Object>> propertiesToJson = new ArrayList<>();
        if (_workflowRightHelper.canRead(workflowDescriptor))
        {
            ActionDescriptor action = workflowDescriptor.getAction(actionId);
            if (action != null)
            {
                propertiesToJson.addAll(_properties2JSON(action.getMetaAttributes()));
            }
        }
        
        return Map.of("data", propertiesToJson);
    }
    
    private List<Map<String, Object>> _properties2JSON(Map<String, Object> metaAttributes)
    {
        List<Map<String, Object>> propertiesToJson = new ArrayList<>();
        
        for (Entry<String, Object> entry : metaAttributes.entrySet())
        {
            Map<String, Object> properties = new HashMap<>();
            properties.put("id", entry.getKey());
            properties.put("value", entry.getValue());
            propertiesToJson.add(properties);
        }
        
        return propertiesToJson;
    }
    
    /**
     * Get the used property names for current workflow element 
     * @param workflowName unique name of current workflow
     * @param stepId id of current step
     * @param actionId id of current action, can be null if only a step is selected
     * @return a set of the properties' names
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Set<String> getPropertiesNames(String workflowName, Integer stepId, Integer actionId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        Map<String, Object> properties = _getProperties(workflowDescriptor, stepId, actionId);
        return properties.keySet();
    }

    /**
     * Add a property to a workflow element
     * @param workflowName unique name of current workflow
     * @param stepId id of current step
     * @param actionId id of current action, can be null if only a step is selected
     * @param name the new name to save
     * @param value the value to save
     * @return the property's infos or an error message
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addProperty(String workflowName, Integer stepId, Integer actionId, String name, String value)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        if (StringUtils.isBlank(name))
        {
            return Map.of("message", "blank-name");
        }
        Map<String, Object> properties = _getProperties(workflowDescriptor, stepId, actionId);
        if (properties.get(name) != null)
        {
            return Map.of("message", "duplicate-name");
        }
        properties.put(name, value);
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getPropertiesInfos(workflowDescriptor, stepId, actionId, name, value, true);
    }

    /**
     * Edit property in a workflow element
     * @param workflowName unique name of current workflow
     * @param stepId id of current step
     * @param actionId id of current action, can be null if only a step is selected
     * @param oldName the old name for this property
     * @param newName the new name to save
     * @param newValue the value to save
     * @return the property's infos or an error message
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editProperty(String workflowName, Integer stepId, Integer actionId, String oldName, String newName, String newValue)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        if (StringUtils.isBlank(newName))
        {
            return Map.of("message", "blank-name");
        }
        
        Map<String, Object> properties = _getProperties(workflowDescriptor, stepId, actionId);
        
        if (StringUtils.isNotBlank(oldName) && !oldName.equals(newName))
        {
            properties.remove(oldName);
        }
        properties.put(newName, newValue);
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getPropertiesInfos(workflowDescriptor, stepId, actionId, newName, newValue, true);
    }
    
    /**
     * Remove a property from a workflow element
     * @param workflowName unique name of current workflow
     * @param stepId id of current step
     * @param actionId id of current action, can be null if only a step is selected
     * @param name the new name to save
     * @return the property's parent infos or an error message
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteProperty(String workflowName, Integer stepId, Integer actionId, String name)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        if (StringUtils.isBlank(name))
        {
            return Map.of("message", "blank-name");
        }
        
        Map<String, Object> properties = _getProperties(workflowDescriptor, stepId, actionId);
        properties.remove(name);
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getPropertiesInfos(workflowDescriptor, stepId, actionId, name, null, true);
    }
    
    private Map<String, Object> _getProperties(WorkflowDescriptor workflowDescriptor, Integer stepId, Integer actionId)
    {
        Map<String, Object> metaAttributes = new HashMap<>();
        
        if (actionId == null)
        {
            StepDescriptor step = workflowDescriptor.getStep(stepId);
            metaAttributes = step.getMetaAttributes();
        }
        else
        {
            ActionDescriptor action = workflowDescriptor.getAction(actionId);
            metaAttributes = action.getMetaAttributes();
        }
        
        return metaAttributes;
    }
    
    private Map<String, Object> _getPropertiesInfos(WorkflowDescriptor workflow, Integer stepId, Integer actionId, String name, String value, boolean hasChanges)
    {
        Map<String, Object> results = new HashMap<>();
        results.put("workflowId", workflow.getName());
        results.put("hasChanges", hasChanges);
        results.put("stepId", stepId);
        results.put("stepLabel", _workflowStepDAO.getStepLabel(workflow, stepId));
        if (actionId != null)
        {
            ActionDescriptor action = workflow.getAction(actionId);
            results.put("actionId", actionId);
            results.put("actionLabel", _workflowTransitionDAO.getActionLabel(action));
        }
        results.put("name", name);
        results.put("value", value);
        return results;
    }
    
}
