/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.workflow.component.WorkflowLanguageManager;
import org.ametys.plugins.workflow.support.I18nHelper;
import org.ametys.plugins.workflow.support.WorflowRightHelper;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowSessionHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.loader.AbstractDescriptor;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.ConditionalResultDescriptor;
import com.opensymphony.workflow.loader.ConditionsDescriptor;
import com.opensymphony.workflow.loader.DescriptorFactory;
import com.opensymphony.workflow.loader.ResultDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * DAO for workflow steps
 */
public class WorkflowStepDAO extends AbstractLogEnabled implements Component, Serviceable
{
    /** The component's role */
    public static final String ROLE =  WorkflowStepDAO.class.getName();
    
    /** Id for initial step */
    public static final String INITIAL_STEP_ID = "step0";
    
    /** The default label for steps */
    public static final I18nizableText DEFAULT_STEP_NAME = new I18nizableText("plugin.workflow", "PLUGIN_WORKFLOW_DEFAULT_STEP_LABEL");
    
    /** Default path for svg step icons */
    private static final String __DEFAULT_SVG_STEP_ICON_PATH = "plugin:cms://resources/img/history/workflow/step_0_16.png";
    
    /** Default path for node step icons */
    private static final String __DEFAULT_STEP_ICON_PATH = "/plugins/cms/resources/img/history/workflow/step_0_16.png";
    
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    /** The helper for i18n translations and catalogs */
    protected I18nHelper _i18nHelper;
    
    /** The workflow session helper */
    protected WorkflowSessionHelper _workflowSessionHelper;
    
    /** The workflow right helper */
    protected WorflowRightHelper _workflowRightHelper;
    
    /** The workflow condition DAO */
    protected WorkflowConditionDAO _workflowConditionDAO;
    
    /** The workflow result DAO */
    protected WorkflowResultDAO _workflowResultDAO;
   
    /** The workflow transition DAO */
    protected WorkflowTransitionDAO _workflowTransitionDAO;
    
    /** The workflow language manager */
    protected WorkflowLanguageManager _workflowLanguageManager;
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _workflowHelper = (WorkflowHelper) smanager.lookup(WorkflowHelper.ROLE);
        _workflowSessionHelper = (WorkflowSessionHelper) smanager.lookup(WorkflowSessionHelper.ROLE);
        _workflowRightHelper = (WorflowRightHelper) smanager.lookup(WorflowRightHelper.ROLE);
        _workflowConditionDAO = (WorkflowConditionDAO) smanager.lookup(WorkflowConditionDAO.ROLE);
        _workflowTransitionDAO = (WorkflowTransitionDAO) smanager.lookup(WorkflowTransitionDAO.ROLE);
        _workflowLanguageManager = (WorkflowLanguageManager) smanager.lookup(WorkflowLanguageManager.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _i18nHelper = (I18nHelper) smanager.lookup(I18nHelper.ROLE);
        _workflowResultDAO = (WorkflowResultDAO) smanager.lookup(WorkflowResultDAO.ROLE);
    }
    
    /**
     * Verify that current workflow has steps
     * @param workflowName the workflow's unique name
     * @return true if worflow has steps
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public boolean hasSteps(String workflowName)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        _workflowRightHelper.checkReadRight(workflowDescriptor);
        return !workflowDescriptor.getSteps().isEmpty();
    }
    
    /**
     * Get the step editable infos
     * @param workflowName current workflow's id
     * @param stepId current step's id
     * @return a map of step infos and non-available ids
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStepInfos(String workflowName, Integer stepId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkReadRight(workflowDescriptor);
        
        Map<String, Object> stepInfos = new HashMap<>();
        List<Integer> stepIds = _getUsedStepIds(workflowDescriptor);
        Map<String, String> translations = new HashMap<>();
        if (stepId == null) //creation mode
        {
            int id = _getUniqueStepId(workflowDescriptor);
            stepInfos.put("id", id);
            translations.put(_workflowLanguageManager.getCurrentLanguage(), _i18nUtils.translate(DEFAULT_STEP_NAME));
        }
        else //edit mode
        {
            stepIds.remove(stepId);
            stepInfos.put("id", stepId);
            I18nizableText labelKey = getStepLabel(workflowDescriptor, stepId);
            translations = _workflowSessionHelper.getTranslation(workflowName, labelKey);
            if (translations == null)
            {
                translations = Map.of(_workflowLanguageManager.getCurrentLanguage(), getStepLabelAsString(workflowDescriptor, stepId, false));
            }
        }
        stepInfos.put("labels", translations);
        stepInfos.put("ids", stepIds);
        
        return stepInfos;
    }

    @SuppressWarnings("unchecked")
    private List<Integer> _getUsedStepIds(WorkflowDescriptor workflowDescriptor)
    {
        List<Integer> usedIds = (List<Integer>) workflowDescriptor.getSteps().stream()
                .map(s -> ((StepDescriptor) s).getId())
                .collect(Collectors.toList());
        usedIds.add(0);
        return usedIds;
    }
    
    /**
     * Create a new step and add it to current workflow
     * @param workflowName current workflow's id
     * @param stepId the new step id
     * @param labels the new step labels
     * @return  map of the step infos
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> createStep(String workflowName, Integer stepId, Map<String, String> labels)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        Map<String, Object> results = new HashMap<>();
        
        List<Integer> stepIds = _getUsedStepIds(workflowDescriptor); //test if new id is unique between workflow's steps
        if (stepIds.contains(stepId) || stepId == 0)
        {
            results.put("message", "duplicate-id");
            return results;
        }
        
        DescriptorFactory factory = new DescriptorFactory();
        StepDescriptor stepDescriptor = factory.createStepDescriptor();
        stepDescriptor.setId(stepId);
        I18nizableText stepLabelKey = _i18nHelper.generateI18nKey(workflowName, "STEP", stepId);
        stepDescriptor.setName(stepLabelKey.toString());
        workflowDescriptor.addStep(stepDescriptor);
        
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        _workflowSessionHelper.updateTranslations(workflowName, stepLabelKey, labels);
        
        results.put("stepId", stepId);
        results.put("stepLabels", labels);
        results.put("workflowId", workflowName);
        
        return results;
    }

    private int _getUniqueStepId(WorkflowDescriptor workflowDescriptor)
    {
        List<Integer> stepIds = _getUsedStepIds(workflowDescriptor);
        int id = 1;
        while (stepIds.contains(id))
        {
            id++;
        }
        return id;
    }
    
    /**
     * Edit the step label
     * @param workflowName current workflow's id
     * @param stepId the step's id
     * @param newMainLabel the new label in the current application's language
     * @return map of the step infos if edit worked, contain error message else
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editStepLabel(String workflowName, Integer stepId, String newMainLabel)
    {
        return editStep(workflowName, stepId, stepId, Map.of(_workflowLanguageManager.getCurrentLanguage(), newMainLabel));
    }
    
    /**
     * Edit the step
     * @param workflowName current workflow's id
     * @param oldId the step's last id
     * @param id the step's new id
     * @param labels the new step labels
     * @return map of the step infos if edit worked, contain error message else
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editStep(String workflowName, Integer oldId, Integer id, Map<String, String> labels)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        Map<String, Object> results = new HashMap<>();
        
        StepDescriptor stepDescriptor = workflowDescriptor.getStep(oldId);
        
        if (id != oldId) //if step id has been edited
        {
            if (!getIncomingActions(oldId, workflowDescriptor).isEmpty()) //edition on id can't happen if there are transitions leading to this step
            {
                results.put("message", "incoming-actions");
                return results;
            }
            List<Integer> stepIds = _getUsedStepIds(workflowDescriptor); //test if new id is unique between workflow's steps
            if (stepIds.contains(id))
            {
                results.put("message", "duplicate-id");
                return results;
            }
            stepDescriptor.setId(id);
        }
        
        String defaultCatalog = _workflowHelper.getWorkflowCatalog(workflowName);
        I18nizableText labelKey = getStepLabel(workflowDescriptor, id);
        if (!defaultCatalog.equals(labelKey.getCatalogue()))
        {
            labelKey = new I18nizableText(defaultCatalog, labelKey.getKey());
            String newName = labelKey.toString();
            stepDescriptor.setName(newName);
        }
        _workflowSessionHelper.updateTranslations(workflowName, labelKey, labels);
        
        results.put("stepId", id);
        results.put("oldStepId", oldId);
        results.put("stepLabels", labels);
        results.put("workflowId", workflowName);
        
        return results;
    }
    
    /**
     * Delete the step from workflow
     * @param workflowName current workflow's id
     * @param stepId current step's id
     * @return an error message if deleting couldn't proceed
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteStep(String workflowName, Integer stepId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        Map<String, Object> results = new HashMap<>();
        
        StepDescriptor stepDescriptor = workflowDescriptor.getStep(stepId);
        
        if (getIncomingActions(stepId, workflowDescriptor).isEmpty()) //we can't delete this step if there are transitions having current step as result
        {
            I18nizableText stepLabel = getStepLabel(workflowDescriptor, stepId);
            workflowDescriptor.getSteps().remove(stepDescriptor);
            _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
            _workflowSessionHelper.removeTranslation(workflowName, getStepLabel(stepDescriptor));
            
            results.put("stepId", stepId);
            results.put("stepLabels", stepLabel);
            results.put("workflowId", workflowName);
        }
        else
        {
            results.put("message", "incoming-actions");
        }
        
        return results;
    }

    /**
     * Get the step label as new I18nizableText
     * @param stepDescriptor the current step
     * @return the step label
     */
    public I18nizableText getStepLabel(StepDescriptor stepDescriptor)
    {
        return new I18nizableText("application", stepDescriptor.getName());
    }
    
    
    /**
     * Get the workflow editor tree's nodes
     * @param currentNode id of the current node
     * @param workflowName unique name of current workflow
     * @return a map of the current node's children
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStepNodes(String currentNode, String workflowName)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkReadRight(workflowDescriptor);
        
        List<Map<String, Object>> nodes = new ArrayList<>();
        boolean canWrite = _workflowRightHelper.canWrite(workflowDescriptor);
        
        if (workflowName != null)
        {
            List<ActionDescriptor> initialActions = workflowDescriptor.getInitialActions();
            if (currentNode.equals("root"))
            {
                //initial step
                Map<String, Object> infosInitialStep = _step2JSON(workflowDescriptor, 0, initialActions.size() > 0, false, false);
                nodes.add(infosInitialStep);
                
                //other steps
                List<StepDescriptor> steps = workflowDescriptor.getSteps();
                for (StepDescriptor step : steps)
                {
                    Map<String, Object> infos = _step2JSON(workflowDescriptor, step.getId(), step.getActions().size() > 0, false, canWrite);
                    nodes.add(infos);
                }
            }
            else if (currentNode.equals(INITIAL_STEP_ID))
            {
                //initial actions
                for (ActionDescriptor initialAction : initialActions)
                {
                    nodes.add(_action2JSON(currentNode, initialAction, workflowDescriptor, canWrite));
                }
            }
            else
            {
                //regular actions
                StepDescriptor step = workflowDescriptor.getStep(Integer.valueOf(currentNode.substring("step".length())));
                for (ActionDescriptor transition : (List<ActionDescriptor>) step.getActions())
                {
                    nodes.add(_action2JSON(currentNode, transition, workflowDescriptor, canWrite));
                }
            }
        }
        
        return Map.of("steps", nodes);
    }
    
    /**
     * Get the action infos for tree panel node
     * @param stepId id of current step  node
     * @param action currently processed action
     * @param workflowDescriptor current workflow
     * @param canWrite true if current user has edition right on current workflow
     * @return map of the action infos
     */
    protected Map<String, Object> _action2JSON(String stepId, ActionDescriptor action, WorkflowDescriptor workflowDescriptor, boolean canWrite)
    {
        Set<StepWithIcon> finalSteps = _getActionFinalSteps(action, workflowDescriptor);
        Map<Integer, Object> finalStepNames = finalSteps.stream().collect(Collectors.toMap(s -> s.id(), s -> s.label()));
        Map<Integer, Object> finalStepIcons = finalSteps.stream().collect(Collectors.toMap(s -> s.id(), s -> s.iconPath()));
        
        String iconPath = _workflowTransitionDAO.getActionIconPath(workflowDescriptor.getName(), action);
        
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", stepId + "-action" + action.getId());
        infos.put("elementId", action.getId());
        infos.put("smallIcon", iconPath);
        infos.put("label", _workflowTransitionDAO.getActionLabel(workflowDescriptor.getName(), action));
        infos.put("elementType", "action");
        infos.put("hasChildren", false);
        infos.put("targetedStepNames", finalStepNames);
        infos.put("targetedStepIcons", finalStepIcons);
        infos.put("canWrite", canWrite);
        
        return infos;
    }
    
    /**
     * Get the conditional and unconditional results of current action
     * @param action the current action
     * @param workflowDescriptor the current workflow
     * @return a list of the final steps as (stepId, stepLabel, StepIconPath)
     */
    protected Set<StepWithIcon> _getActionFinalSteps(ActionDescriptor action, WorkflowDescriptor workflowDescriptor)
    {
        Set<StepWithIcon> steps = new HashSet<>();
        for (StepDescriptor step : getOutgoingSteps(action, workflowDescriptor))
        {
            int stepId = step.getId();
            steps.add(new StepWithIcon(
                stepId,
                getStepLabelAsString(workflowDescriptor, stepId, false),
                getStepIconPath(workflowDescriptor, stepId))
            );
        }
        
        return steps;
    }
    
    /**
     * Get possible outgoing steps for action
     * @param action the current action
     * @param workflowDescriptor the current workflow
     * @return a set of the outgoing steps
     */
    public Set<StepDescriptor> getOutgoingSteps(ActionDescriptor action, WorkflowDescriptor workflowDescriptor)
    {
        Set<StepDescriptor> outgoingSteps = new HashSet<>();
        ResultDescriptor unconditionalResult = action.getUnconditionalResult();
        boolean hasSameStepTarget = false;
        if (unconditionalResult.getStep() != -1)
        {
            StepDescriptor unconditionalStep = workflowDescriptor.getStep(unconditionalResult.getStep());
            outgoingSteps.add(unconditionalStep);
        }
        else
        {
            hasSameStepTarget = true;
        }
        List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
        for (ConditionalResultDescriptor result : conditionalResults)
        {
            StepDescriptor conditionalStep = workflowDescriptor.getStep(result.getStep());
            if (conditionalStep != null)
            {
                outgoingSteps.add(conditionalStep);
            }
            else
            {
                hasSameStepTarget = true;
            }
        }
        if (hasSameStepTarget)
        {
            outgoingSteps.addAll(getIncomingSteps(action.getId(), workflowDescriptor));
        }
        
        return outgoingSteps;
    }
    
    /**
     * Get possible incoming steps for action
     * @param actionId the current action's id
     * @param workflowDescriptor the current  workflow
     * @return a set of the action's incoming steps
     */
    public Set<StepDescriptor> getIncomingSteps(int actionId, WorkflowDescriptor workflowDescriptor)
    {
        Set<StepDescriptor> incomingSteps = new HashSet<>();
        List<StepDescriptor> steps = workflowDescriptor.getSteps();
        for (StepDescriptor step : steps)
        {
            if (step.getAction(actionId) != null)
            {
                incomingSteps.add(step);
            }
        }
        return incomingSteps;
    }
    
    /**
     * Get current action's final steps and associated conditions
     * @param currentNode id of current node
     * @param workflowName unique name of current workflow
     * @param actionId id of current action
     * @return a map of current node's children
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getFinalSteps(String currentNode, String workflowName, Integer actionId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        List<Map<String, Object>> nodes = new ArrayList<>();
        if (_workflowRightHelper.canRead(workflowDescriptor))
        {
            ActionDescriptor action = workflowDescriptor.getAction(actionId);
            boolean canWrite = _workflowRightHelper.canWrite(workflowDescriptor);
            
            List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
            
            if (currentNode.equals("root")) //get results(steps) nodes
            {
                ResultDescriptor unconditionalResult = action.getUnconditionalResult();
                int stepId = unconditionalResult.getStep();
                Map<String, Object> step2json = _step2JSON(workflowDescriptor, stepId, false, stepId != -1, canWrite);
                step2json.put("isConditional", false);
                nodes.add(step2json);
                
                for (ConditionalResultDescriptor result : conditionalResults)
                {
                    stepId = result.getStep();
                    Map<String, Object> conditionalStep2Json = _step2JSON(workflowDescriptor, stepId, !result.getConditions().isEmpty(), stepId != -1, canWrite);
                    conditionalStep2Json.put("isConditional", true);
                    nodes.add(conditionalStep2Json);
                }
            }
            else //get conditions nodes,
            {
                //conditions to display
                List<AbstractDescriptor> conditions = _workflowResultDAO.getChildrenResultConditions(currentNode, action, conditionalResults);
                if (!conditions.isEmpty())
                {
                    String[] path = _workflowResultDAO.getPath(currentNode);
                    int stepId = Integer.valueOf(path[0].substring(4));
                    ConditionsDescriptor rootOperator = _workflowResultDAO.getRootResultConditions(conditionalResults, stepId).get(0);
                    boolean rootIsAND = !rootOperator.getType().equals(WorkflowConditionDAO.OR);
                    for (int i = 0; i < conditions.size(); i++)
                    {
                        nodes.add(_workflowConditionDAO.conditionToJSON(conditions.get(i), currentNode, i, rootIsAND));
                    }
                }
            }
        }
        
        return Map.of("results", nodes);
    }
    
    /**
     * Get step infos
     * @param workflowDescriptor current workflow
     * @param stepId id of current step
     * @param hasChildren true if step has actions
     * @param showId true if id needs to be displayed in the label
     * @param canWrite true if current user has edition right on current workflow
     * @return a map of the step infos
     */
    protected Map<String, Object> _step2JSON(WorkflowDescriptor workflowDescriptor, int stepId, boolean hasChildren, boolean showId, boolean canWrite)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", "step" + stepId);
        infos.put("elementId", stepId);
        infos.put("label", getStepLabelAsString(workflowDescriptor, stepId, showId));
        infos.put("elementType", "step");
        infos.put("hasChildren", hasChildren);
        infos.put("canWrite", canWrite);
        try
        {
            infos.put("smallIcon", getStepIconPath(workflowDescriptor, stepId));
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred while getting icon path for step id {}", stepId, e);
        }
       
        return infos;
    }
    
    /**
     * Get the workflow's steps available as unconditional result for actions
     * @param workflowName the current workflow name
     * @param actionId id of current action if exist, can be null
     * @param isInitialState true if current selected state is the initial state
     * @return a map of the workflow steps
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStatesToJson(String workflowName, Integer actionId, Boolean isInitialState)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        _workflowRightHelper.checkReadRight(workflowDescriptor);
        List<Map<String, Object>> states = new ArrayList<>();
        List<Integer> conditionalStepIds = new ArrayList<>();
        List<StepDescriptor> steps = workflowDescriptor.getSteps();
        if (actionId != null)
        {
            ActionDescriptor action = workflowDescriptor.getAction(actionId);
            List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
            for (ConditionalResultDescriptor conditionalResult : conditionalResults)
            {
                conditionalStepIds.add(conditionalResult.getStep());
            }
        }
        for (StepDescriptor step : steps)
        {
            int stepId = step.getId();
            if (!conditionalStepIds.contains(stepId))
            {
                Map<String, Object> stateInfos = new HashMap<>();
                stateInfos.put("id", stepId);
                stateInfos.put("label", getStepLabelAsString(workflowDescriptor, stepId, true));
                states.add(stateInfos);
            }
        }
        if (!isInitialState && !conditionalStepIds.contains(-1))
        {
            //Same state
            Map<String, Object> stateInfos = new HashMap<>();
            stateInfos.put("id", -1);
            stateInfos.put("label", new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_RESULTS_SAME_STEP"));
            states.add(stateInfos);
        }
        
        return Map.of("data", states);
    }
    
    
    /**
     * Get the translated step label
     * @param workflowDescriptor current workflow
     * @param stepId id of current step
     * @param showId true if id needs to be displayed in the label
     * @return the step label as string
     */
    public String getStepLabelAsString(WorkflowDescriptor workflowDescriptor, int stepId, boolean showId)
    {
        I18nizableText label = getStepLabel(workflowDescriptor, stepId);
        return showId
                ? _i18nHelper.translateKey(workflowDescriptor.getName(), label, DEFAULT_STEP_NAME) + " (" + stepId + ")"
                : _i18nHelper.translateKey(workflowDescriptor.getName(), label, DEFAULT_STEP_NAME);
    }
    
    /**
     * Get the step's icon path
     * @param workflowDescriptor current worklfow
     * @param stepId id of current step
     * @return the icon path
     */
    public String getStepIconPath(WorkflowDescriptor workflowDescriptor, int stepId)
    {
        I18nizableText label = getStepLabel(workflowDescriptor, stepId);
        label = _workflowSessionHelper.getOldLabelKeyIfCloned(workflowDescriptor.getName(), label);
        return _workflowHelper.getElementIconPath(label, __DEFAULT_STEP_ICON_PATH);
    }
    
    /**
     * Get the step's icon path as base 64 for svg links
     * @param workflowDescriptor current worklfow
     * @param stepId id of current step
     * @return the icon path as base 64
     */
    public String getStepIconPathAsBase64(WorkflowDescriptor workflowDescriptor, int stepId)
    {
        I18nizableText label = getStepLabel(workflowDescriptor, stepId);
        label = _workflowSessionHelper.getOldLabelKeyIfCloned(workflowDescriptor.getName(), label);
        return _workflowHelper.getElementIconAsBase64(label, __DEFAULT_SVG_STEP_ICON_PATH);
    }
    
    /**
     * Get the step i18n label
     * @param workflowDescriptor current workflow
     * @param stepId id of current step
     * @return the i18n step label
     */
    public I18nizableText getStepLabel(WorkflowDescriptor workflowDescriptor, int stepId)
    {
        switch (stepId)
        {
            case -1:
                return new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_RESULTS_SAME_STEP");
            case 0:
                return new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_INITIAL_STEP_NAME");
            default:
                StepDescriptor step = workflowDescriptor.getStep(stepId);
                return getStepLabel(step);
        }
    }
    
    /**
     * Get a list of actions outgoing from current step
     * @param stepId id of current step
     * @param workflow current workflow
     * @return the list of outgoing actions
     */
    public List<ActionDescriptor> getOutgoingActions(int stepId, WorkflowDescriptor workflow)
    {
        return stepId != 0
                ? workflow.getStep(stepId).getActions()
                : workflow.getInitialActions();
    }
    
    /**
     * Get a set of actions incoming to current step
     * @param stepId id of current step
     * @param workflow current workflow
     * @return the set of outgoing actions
     */
    @SuppressWarnings("unchecked")
    public Set<ActionDescriptor> getIncomingActions(int stepId , WorkflowDescriptor workflow)
    {
        Set<ActionDescriptor> incomingActions = new HashSet<>();
        if (stepId != 0)
        {
            incomingActions.addAll(_getIncomingActionsFromList(stepId, workflow.getInitialActions()));
            List<StepDescriptor> steps = workflow.getSteps();
            for (StepDescriptor otherSteps : steps)
            {
                if (otherSteps.getId() != stepId)
                {
                    List<ActionDescriptor> actions = otherSteps.getActions();
                    incomingActions.addAll(_getIncomingActionsFromList(stepId, actions));
                }
            }
        }
        return incomingActions;
    }
    
    /**
     * Get a set of incoming actions if present in actions list
     * @param stepId id of current step
     * @param actions list of other step's actions
     * @return a list containing other step's outgoing actions that are incoming to current step
     */
    protected Set<ActionDescriptor> _getIncomingActionsFromList(int stepId, List<ActionDescriptor> actions)
    {
        Set<ActionDescriptor> incoming = new HashSet<>();
        for (ActionDescriptor action : actions)
        {
            ResultDescriptor unconditionalResult = action.getUnconditionalResult();
            if (unconditionalResult.getStep() == stepId)
            {
                incoming.add(action);
            }
            else
            {
                boolean leadToStep = false;
                List<ResultDescriptor> conditionalResults = action.getConditionalResults();
                int indexResult = 0;
                while (!leadToStep && indexResult < conditionalResults.size())
                {
                    if (conditionalResults.get(indexResult).getStep() == stepId)
                    {
                        incoming.add(action);
                        leadToStep = true;
                    }
                    indexResult++;
                }
            }
        }
        return incoming;
    }
    
    /**
     * Get id of the first step having current action, INITIAL_STEP_ID if current action is an initial action
     * @param stepId id of current step
     * @param steps list of all the steps in current workflow
     * @param actionId id of current action
     * @return the id of the first found step having current action
     */
    public String getFirstParentStepId(int stepId, List<StepDescriptor> steps, Integer actionId)
    {
        String firstParentStepId = "";
        int i = 0;
        do
        {
            StepDescriptor otherStep = steps.get(i);
            if (otherStep.getId() != stepId && otherStep.getAction(actionId) != null)
            {
                firstParentStepId = String.valueOf(otherStep.getId());
            }
            i++;
        } while (firstParentStepId.isEmpty() && i < steps.size());
        return firstParentStepId.isBlank() ? "0" : firstParentStepId;
    }
    
    private record StepWithIcon(Integer id, String label, String iconPath) { /* empty */ }
}
