/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.definition;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.plugins.workflow.support.I18nHelper;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

import com.opensymphony.workflow.loader.WorkflowDescriptor;
import com.opensymphony.workflow.loader.WorkflowLoader;

/**
 * Object to describe a workflow.
 */
public class WorkflowDefinition extends AbstractLogEnabled implements Component, Serviceable, Configurable, PluginAware
{
    /** The source resolver */
    protected SourceResolver _srcResolver;
    
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    private String _configuredFile;
    private I18nizableText _label;
    private WorkflowDescriptor _descriptor;
    private String _id;
    private String _pluginName;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _configuredFile = configuration.getChild("file").getValue(null);
        
        String defaultCatalog = getDefaultCatalog();
        I18nizableText defaultValue = new I18nizableText(defaultCatalog, I18nHelper.buildI18nWorkflowKey(_id));
        _label = I18nizableText.parseI18nizableText(configuration.getChild("label"), defaultCatalog, defaultValue);
        // Lazy loaded
        _descriptor = null;
    }
    
    /**
     * Get the default catalog for the workflow name.
     * @return the default catalog
     */
    public String getDefaultCatalog()
    {
        return "plugin." + _pluginName;
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
        _pluginName = pluginName;
    }
    
    /**
     * Get the workflow label.
     * @return The workflow label, default is plugin.[pluginName]:WORKFLOW_[workflowNameUpperCased]
     */
    public I18nizableText getLabel()
    {
        return _label;
    }
    
    /**
     * Get the location of the workflow file.
     * @return the location of the workflow file
     */
    public String getLocation()
    {
        return _getFilePathFromParam(_id)                                   // Search in WEB-INF/param/workflows/[workflowId].xml
                .or(() -> Optional.ofNullable(_configuredFile))             // Then in the path defined by the "file" child of the configuration if defined
                .orElseGet(() -> _getFilePathFromPlugin(_pluginName, _id)); // Then in [plugin]/workflows/[workflowId].xml
    }
    
    /**
     * Get or load the workflow descriptor.
     * @param validate if the descriptor has to be valid
     * @return the workflow descriptor
     * @throws Exception if an error occurs
     */
    public WorkflowDescriptor getDescriptor(boolean validate) throws Exception
    {
        if (_descriptor == null)
        {
            try (InputStream is = getDefinition())
            {
                _descriptor = WorkflowLoader.load(is, validate);
            }
            
            _descriptor.setName(_id);
        }
        
        return _descriptor;
    }
    
    /**
     * Get the workflow definition
     * @return the workflow definition
     * @throws IOException if an error occurred
     */
    protected InputStream getDefinition() throws IOException
    {
        String location = getLocation();
        getLogger().debug("Loading '{}' for the workflow '{}'.", location, _id);
        
        Source source = null;
        
        try
        {
            source = _srcResolver.resolveURI(location);
            return source.getInputStream();
        }
        finally
        {
            _srcResolver.release(source);
        }
        
    }
    
    /**
     * Method to reset the workflow.
     */
    public void reset()
    {
        _descriptor = null;
    }
    
    private Optional<String> _getFilePathFromParam(String workflowId)
    {
        return Optional.of(workflowId)
                .map(id -> id + ".xml")
                .map(filename -> new File(_workflowHelper.getParamWorkflowDir(), filename))
                .filter(File::exists)
                .filter(File::isFile)
                // Transform it to URI with context://
                .map(file -> "context://WEB-INF/param/workflows/" + file.getName());
    }
    
    private String _getFilePathFromPlugin(String pluginName, String workflowId)
    {
        return "plugin:" + pluginName + "://workflows/" + workflowId + ".xml";
    }
}
