/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.store;

import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;

import org.ametys.plugins.repository.AmetysRepositoryException;

import com.opensymphony.workflow.spi.Step;

/**
 * OSWorkflow step with additional properties.<br>
 */
public class AmetysStep implements Step
{
    /** Custom property prefix. */
    protected static final String __CUSTOM_PROP_PREFIX = "ametys:";
    
    private Node _node;
    private AbstractJackrabbitWorkflowStore _store;
    
    /**
     * Build an ametys step.
     * @param node the backing JCR node.
     * @param store the workflow store.
     */
    public AmetysStep(Node node, AbstractJackrabbitWorkflowStore store)
    {
        this._node = node;
        this._store = store;
    }
    
    /**
     * Get the backing JCR node.
     * @return the backing JCR node.
     */
    Node getNode()
    {
        return _node;
    }
    
    @Override
    public long getId()
    {
        try
        {
            return _node.getProperty(AbstractJackrabbitWorkflowStore.__ID_PROPERTY).getLong();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the ID property.", e);
        }
    }
    
    @Override
    public long getEntryId()
    {
        try
        {
            return _node.getParent().getProperty(AbstractJackrabbitWorkflowStore.__ID_PROPERTY).getLong();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the entry ID property.", e);
        }
    }
    
    @Override
    public int getStepId()
    {
        try
        {
            return (int) _node.getProperty(AbstractJackrabbitWorkflowStore.__STEP_ID_PROPERTY).getLong();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the step ID property.", e);
        }
    }
    
    /**
     * Set the step ID.
     * @param stepId the step ID to set.
     */
    public void setStepId(int stepId)
    {
        try
        {
            _node.setProperty(AbstractJackrabbitWorkflowStore.__STEP_ID_PROPERTY, stepId);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the step ID property.", e);
        }
    }
    
    @Override
    public int getActionId()
    {
        try
        {
            if (_node.hasProperty(AbstractJackrabbitWorkflowStore.__ACTION_ID_PROPERTY))
            {
                return (int) _node.getProperty(AbstractJackrabbitWorkflowStore.__ACTION_ID_PROPERTY).getLong();
            }
            
            return 0;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the action ID property.", e);
        }
    }
    
    /**
     * Set the action ID.
     * @param actionId the action ID to set.
     */
    public void setActionId(int actionId)
    {
        try
        {
            _node.setProperty(AbstractJackrabbitWorkflowStore.__ACTION_ID_PROPERTY, actionId);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the action ID property.", e);
        }
    }
    
    @Override
    public String getCaller()
    {
        try
        {
            String caller = null;
            
            if (_node.hasProperty(AbstractJackrabbitWorkflowStore.__CALLER_PROPERTY))
            {
                caller = _node.getProperty(AbstractJackrabbitWorkflowStore.__CALLER_PROPERTY).getString();
            }
            
            return caller;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the caller property.", e);
        }
    }
    
    /**
     * Set the caller.
     * @param caller the caller to set.
     */
    public void setCaller(String caller)
    {
        try
        {
            _node.setProperty(AbstractJackrabbitWorkflowStore.__CALLER_PROPERTY, caller);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the caller property.", e);
        }
    }
    
    @Override
    public Date getStartDate()
    {
        try
        {
            Date value = null;
            
            if (_node.hasProperty(AbstractJackrabbitWorkflowStore.__START_DATE_PROPERTY))
            {
                value = _node.getProperty(AbstractJackrabbitWorkflowStore.__START_DATE_PROPERTY).getDate().getTime();
            }
            
            return value;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the start date property.", e);
        }
    }
    
    /**
     * Set the step start date.
     * @param startDate the start date to set.
     */
    public void setStartDate(Date startDate)
    {
        try
        {
            GregorianCalendar cal = new GregorianCalendar();
            cal.setTime(startDate);
            
            _node.setProperty(AbstractJackrabbitWorkflowStore.__START_DATE_PROPERTY, cal);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the start date property.", e);
        }
    }
    
    @Override
    public Date getDueDate()
    {
        try
        {
            Date value = null;
            
            if (_node.hasProperty(AbstractJackrabbitWorkflowStore.__DUE_DATE_PROPERTY))
            {
                value = _node.getProperty(AbstractJackrabbitWorkflowStore.__DUE_DATE_PROPERTY).getDate().getTime();
            }
            
            return value;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the due date property.", e);
        }
    }
    
    /**
     * Set the step due date.
     * @param dueDate the due date to set.
     */
    public void setDueDate(Date dueDate)
    {
        try
        {
            GregorianCalendar cal = new GregorianCalendar();
            cal.setTime(dueDate);
            
            _node.setProperty(AbstractJackrabbitWorkflowStore.__DUE_DATE_PROPERTY, cal);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the due date property.", e);
        }
    }
    
    @Override
    public Date getFinishDate()
    {
        try
        {
            Date value = null;
            
            if (_node.hasProperty(AbstractJackrabbitWorkflowStore.__FINISH_DATE_PROPERTY))
            {
                value = _node.getProperty(AbstractJackrabbitWorkflowStore.__FINISH_DATE_PROPERTY).getDate().getTime();
            }
            
            return value;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the finish date property.", e);
        }
    }
    
    /**
     * Set the step finish date.
     * @param finishDate the finish date to set.
     */
    public void setFinishDate(Date finishDate)
    {
        try
        {
            GregorianCalendar cal = new GregorianCalendar();
            cal.setTime(finishDate);
            
            _node.setProperty(AbstractJackrabbitWorkflowStore.__FINISH_DATE_PROPERTY, cal);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the finish date property.", e);
        }
    }
    
    @Override
    public String getOwner()
    {
        try
        {
            String owner = null;
            
            if (_node.hasProperty(AbstractJackrabbitWorkflowStore.__OWNER_PROPERTY))
            {
                owner = _node.getProperty(AbstractJackrabbitWorkflowStore.__OWNER_PROPERTY).getString();
            }
            
            return owner;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the owner property.", e);
        }
    }
    
    /**
     * Set the owner.
     * @param owner the owner to set.
     */
    public void setOwner(String owner)
    {
        try
        {
            _node.setProperty(AbstractJackrabbitWorkflowStore.__OWNER_PROPERTY, owner);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the owner property.", e);
        }
    }
    
    @Override
    public long[] getPreviousStepIds()
    {
        try
        {
            List<Long> previousStepsIds = new ArrayList<>();
            
            if (_node.hasProperty(AbstractJackrabbitWorkflowStore.__PREVIOUS_STEPS_PROPERTY))
            {
                Value[] previousSteps = _node.getProperty(AbstractJackrabbitWorkflowStore.__PREVIOUS_STEPS_PROPERTY).getValues();
    
                for (Value previousStep : previousSteps)
                {
                    Node historyStep = _node.getSession().getNodeByIdentifier(previousStep.getString());
    
                    long previousStepId = historyStep.getProperty(AbstractJackrabbitWorkflowStore.__ID_PROPERTY).getLong();
                    previousStepsIds.add(previousStepId);
                }
            }
            
            long[] previousIds = new long[previousStepsIds.size()];
            
            for (int i = 0; i < previousIds.length; i++)
            {
                previousIds[i] = previousStepsIds.get(i);
            }
            
            return previousIds;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the previous IDs property.", e);
        }
    }
    
    /**
     * Set the previous step IDs.
     * @param previousStepIds the previous step IDs to set.
     */
    public void setPreviousStepIds(long[] previousStepIds)
    {
        try
        {
            Session session = _node.getSession();
            ValueFactory valueFactory = session.getValueFactory();
            
            Value[] previousStepsRefs = new Value[0];
            
            if (previousStepIds != null)
            {
                previousStepsRefs = new Value[previousStepIds.length];
                
                for (int i = 0; i < previousStepIds.length; i++)
                {
                    long previousStepId = previousStepIds[i];
                    
                    // Retrieve existing step for this entry
                    Node previousStep = _store.getHistoryStepNode(session, getEntryId(), previousStepId);
                    
                    // Use the ValueFactory to set a reference to this step
                    previousStepsRefs[i] = valueFactory.createValue(previousStep);
                }
            }
            
            _node.setProperty(AbstractJackrabbitWorkflowStore.__PREVIOUS_STEPS_PROPERTY, previousStepsRefs);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the previous IDs property.", e);
        }
    }
    
    @Override
    public String getStatus()
    {
        try
        {
            String status = null;
            
            if (_node.hasProperty(AbstractJackrabbitWorkflowStore.__STATUS_PROPERTY))
            {
                status = _node.getProperty(AbstractJackrabbitWorkflowStore.__STATUS_PROPERTY).getString();
            }
            
            return status;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the status property.", e);
        }
    }
    
    /**
     * Set the status.
     * @param status the status to set.
     */
    public void setStatus(String status)
    {
        try
        {
            _node.setProperty(AbstractJackrabbitWorkflowStore.__STATUS_PROPERTY, status);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the status property.", e);
        }
    }
    
    /**
     * Get a custom property value.
     * @param name the property name.
     * @return the custom property value.
     */
    public Object getProperty(String name)
    {
        try
        {
            Object value = null;
            String qName = __CUSTOM_PROP_PREFIX + name;
            
            if (_node.hasProperty(qName))
            {
                Property property = _node.getProperty(qName);
                if (property.getDefinition().isMultiple())
                {
                    value = getMultiValuedProperty(property);
                }
                else
                {
                    value = getSingleValuedProperty(property);
                }
            }
            
            return value;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the custom property: " + name, e);
        }
    }
    
    /**
     * Set a custom property value.
     * @param name the property name.
     * @param value the value to set.
     */
    public void setProperty(String name, Object value)
    {
        try
        {
            ValueFactory valueFactory = _node.getSession().getValueFactory();
            String qName = __CUSTOM_PROP_PREFIX + name;
            
            if (value instanceof Boolean)
            {
                _node.setProperty(qName, (Boolean) value);
            }
            else if (value instanceof boolean[])
            {
                boolean[] v = (boolean[]) value;
                Value[] values = new Value[v.length];
                for (int i = 0; i < v.length; i++)
                {
                    values[i] = valueFactory.createValue(v[i]);
                }
                
                _node.setProperty(qName, values);
            }
            else if (value instanceof Date)
            {
                GregorianCalendar gc = new GregorianCalendar();
                gc.setTime((Date) value);
                _node.setProperty(qName, gc);
            }
            else if (value instanceof Date[])
            {
                Date[] v = (Date[]) value;
                Value[] values = new Value[v.length];
                for (int i = 0; i < v.length; i++)
                {
                    GregorianCalendar gc = new GregorianCalendar();
                    gc.setTime(v[i]);
                    values[i] = valueFactory.createValue(gc);
                }
                
                _node.setProperty(qName, values);
            }
            else if (value instanceof Long)
            {
                _node.setProperty(qName, (Long) value);
            }
            else if (value instanceof long[])
            {
                long[] v = (long[]) value;
                Value[] values = new Value[v.length];
                for (int i = 0; i < v.length; i++)
                {
                    values[i] = valueFactory.createValue(v[i]);
                }
                
                _node.setProperty(qName, values);
            }
            else if (value instanceof String[])
            {
                String[] v = (String[]) value;
                Value[] values = new Value[v.length];
                for (int i = 0; i < v.length; i++)
                {
                    values[i] = valueFactory.createValue(v[i]);
                }
                
                _node.setProperty(qName, values);
            }
            else
            {
                _node.setProperty(qName, value.toString());
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the custom property: " + name, e);
        }
    }
    
    /**
     * Get all the custom property names.
     * @return the custom property names.
     */
    public Set<String> getPropertyNames()
    {
        try
        {
            Set<String> propertyNames = new LinkedHashSet<>();
            
            PropertyIterator properties = _node.getProperties(__CUSTOM_PROP_PREFIX + "*");
            while (properties.hasNext())
            {
                String qName = properties.nextProperty().getName();
                String name = qName.substring(__CUSTOM_PROP_PREFIX.length());
                propertyNames.add(name);
            }
            
            return propertyNames;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error retrieving the custom property list.", e);
        }
    }
    
    /**
     * Save the step.
     */
    public void save()
    {
        try
        {
            _node.getSession().save();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error saving the .", e);
        }
    }
    
    /**
     * Get a single-valued property.
     * @param property the JCR property.
     * @return the property value.
     * @throws RepositoryException if an error occurs.
     */
    protected Object getSingleValuedProperty(Property property) throws RepositoryException
    {
        Object value;
        
        switch (property.getType())
        {
            case PropertyType.BOOLEAN:
                value = property.getBoolean();
                break;
            case PropertyType.DATE:
                value = property.getDate().getTime();
                break;
            case PropertyType.LONG:
                value = property.getLong();
                break;
            default:
                value = property.getString();
                break;
        }
        
        return value;
    }
    
    /**
     * Get a multi-valued property.
     * @param property the JCR property.
     * @return the property value (as an array).
     * @throws RepositoryException if an error occurs.
     */
    protected Object getMultiValuedProperty(Property property) throws RepositoryException
    {
        Object value;
        Value[] values = property.getValues();
        
        switch (property.getType())
        {
            case PropertyType.BOOLEAN:
                boolean[] booleanResults = new boolean[values.length];
                
                for (int i = 0; i < values.length; i++)
                {
                    booleanResults[i] = values[i].getBoolean();
                }
                
                value = booleanResults;
                break;
            case PropertyType.DATE:
                Date[] dateResults = new Date[values.length];
                
                for (int i = 0; i < values.length; i++)
                {
                    dateResults[i] = values[i].getDate().getTime();
                }
                
                value = dateResults;
                break;
            case PropertyType.LONG:
                long[] longResults = new long[values.length];
                
                for (int i = 0; i < values.length; i++)
                {
                    longResults[i] = values[i].getLong();
                }
                
                value = longResults;
                break;
            default:
                String[] stringResults = new String[values.length];
                
                for (int i = 0; i < values.length; i++)
                {
                    stringResults[i] = values[i].getString();
                }
                
                value = stringResults;
                break;
        }
        
        return value;
    }
    
    @Override
    public String toString()
    {
        return "AmetysStep (" + getId() + ")";
    }
    
}
