/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.store;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.datasource.SQLDataSourceManager;
import org.ametys.plugins.workflow.JDBCPropertySet;
import org.ametys.plugins.workflow.PropertySetExpression;
import org.ametys.runtime.config.Config;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.StoreException;
import com.opensymphony.workflow.query.Expression;
import com.opensymphony.workflow.query.NestedExpression;
import com.opensymphony.workflow.query.WorkflowExpressionQuery;
import com.opensymphony.workflow.spi.jdbc.JDBCWorkflowStore;

/**
 * Ametys JDBC workflow store
 */
public class JdbcWorkflowStore extends JDBCWorkflowStore implements AmetysWorkflowStore, Component, Serviceable, Configurable, Initializable, Disposable
{
    /** Avalon role. */
    public static final String ROLE = JdbcWorkflowStore.class.getName();
    
    /** Logger */
    protected Logger _logger = LoggerFactory.getLogger(JdbcWorkflowStore.class);
    
    /** The manager for SQL data source */
    protected SQLDataSourceManager _sqlDataSourceManager;
    
    /** Configured datasource id */
    protected String _dataSourceId;
    
    /** Configured data source type */
    protected String _databaseType;
    
    /** from com.opensymphony.workflow.spi.jdbc.MySQLWorkflowStore */
    protected String _entrySequenceIncrement;
    /** from com.opensymphony.workflow.spi.jdbc.MySQLWorkflowStore */
    protected String _entrySequenceRetrieve;
    /** from com.opensymphony.workflow.spi.jdbc.MySQLWorkflowStore */
    protected String _stepSequenceIncrement;
    /** from com.opensymphony.workflow.spi.jdbc.MySQLWorkflowStore */
    protected String _stepSequenceRetrieve;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _sqlDataSourceManager = (SQLDataSourceManager) manager.lookup(SQLDataSourceManager.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        Configuration dataSourceConf = configuration.getChild("datasource", true);
        String dataSourceConfParam = dataSourceConf.getValue();
        String dataSourceConfType = dataSourceConf.getAttribute("type", "config");
        
        if (Strings.CS.equals(dataSourceConfType, "config"))
        {
            _dataSourceId = Config.getInstance().getValue(dataSourceConfParam);
        }
        else // expecting type="id"
        {
            _dataSourceId = dataSourceConfParam;
        }
    }
    
    public void initialize() throws Exception
    {
        ds = _sqlDataSourceManager.getSQLDataSource(_dataSourceId);
        
        if (ds == null)
        {
            throw new ConfigurationException("Invalid datasource id: " + _dataSourceId);
        }
        
        this.init(Collections.EMPTY_MAP);
    }
    
    public void dispose()
    {
        _dataSourceId = null;
        ds = null;
    }
    
    @Override
    public void init(Map props) throws StoreException
    {
        // Copier les bons paramètres pour JDBCWorkflowStore
        entryTable = "OS_WFENTRY";
        entryId = "ID";
        entryName = "NAME";
        entryState = "STATE";
        historyTable = "OS_HISTORYSTEP";
        historyPrevTable = "OS_HISTORYSTEP_PREV";
        currentTable = "OS_CURRENTSTEP";
        currentPrevTable = "OS_CURRENTSTEP_PREV";
        stepId = "ID";
        stepEntryId = "ENTRY_ID";
        stepStepId = "STEP_ID";
        stepActionId = "ACTION_ID";
        stepOwner = "OWNER";
        stepCaller = "CALLER";
        stepStartDate = "START_DATE";
        stepFinishDate = "FINISH_DATE";
        stepDueDate = "DUE_DATE";
        stepStatus = "STATUS";
        stepPreviousId = "PREVIOUS_ID";
        
        String databaseType;
        try
        {
            databaseType = getDatabaseType();
        }
        catch (SQLException e)
        {
            throw new StoreException("Unable to retrieve database type", e);
        }
        
        // DATABASE_MYSQL
        if (ConnectionHelper.DATABASE_MYSQL.equals(databaseType))
        {
            _entrySequenceIncrement = "INSERT INTO OS_ENTRYIDS (ID) values (null)";
            _entrySequenceRetrieve = "SELECT LAST_INSERT_ID()";
            _stepSequenceIncrement = "INSERT INTO OS_STEPIDS (ID) values (null)";
            _stepSequenceRetrieve = "SELECT LAST_INSERT_ID()";
        }
        
        // DATABASE_DERBY
        else if (ConnectionHelper.DATABASE_DERBY.equals(databaseType))
        {
            _entrySequenceIncrement = "INSERT INTO OS_ENTRYIDS values (DEFAULT)";
            _entrySequenceRetrieve = "VALUES IDENTITY_VAL_LOCAL()";
            _stepSequenceIncrement = "INSERT INTO OS_STEPIDS values (DEFAULT)";
            _stepSequenceRetrieve = "VALUES IDENTITY_VAL_LOCAL()";
        }
        
        // DATABASE_POSTGRES
        else if (ConnectionHelper.DATABASE_POSTGRES.equals(databaseType))
        {
            entrySequence = "SELECT nextval('seq_os_wfentry')";
            stepSequence = "SELECT nextval('seq_os_currentsteps')";
        }
        
        // DATABASE_ORACLE
        else if (ConnectionHelper.DATABASE_ORACLE.equals(databaseType))
        {
            entrySequence = "SELECT seq_os_wfentry.nextval from dual";
            stepSequence = "SELECT seq_os_currentsteps.nextval from dual";
        }
        
        // DATABASE_HSQLDB
        else if (ConnectionHelper.DATABASE_HSQLDB.equals(databaseType))
        {
            entrySequence = "call next value for seq_os_wfentry";
            stepSequence = "call next value for seq_os_currentsteps";
        }
        
        else
        {
            throw new IllegalArgumentException(String.format("Unsupported database type '%s'", databaseType));
        }
    }
    
    /**
     * Database type getter
     * @return The database type
     * @throws SQLException on error
     */
    public String getDatabaseType() throws SQLException
    {
        if (_databaseType == null)
        {
            try (Connection connection = getConnection();)
            {
                _databaseType = ConnectionHelper.getDatabaseType(connection);
            }
        }
        
        return _databaseType;
    }
    
    @Override
    public PropertySet getPropertySet(long id)
    {
        // Utiliser notre propre property set (cocoon connection pool)
        Map<String, String> args = new HashMap<>(1);
        args.put("globalKey", JDBCPropertySet.OSWF_PREFIX + id);
        
        Map<String, Object> config = new HashMap<>(1);
        config.put("datasource", ds);
        
        PropertySet ps = new JDBCPropertySet();
        ps.init(config, args);
        return ps;
    }
    
    @Override
    public boolean shouldClearHistory()
    {
        return true;
    }
    
    @Override
    protected void cleanup(Connection connection, Statement statement, ResultSet result)
    {
        ConnectionHelper.cleanup(result);
        ConnectionHelper.cleanup(statement);
        if (closeConnWhenDone)
        {
            ConnectionHelper.cleanup(connection);
        }
    }

    @Override
    public void deleteInstance(long idWorkflow)
    {
        Connection c = null;
        PreparedStatement stmt = null;

        try
        {
            // Récupérer la connexion du pool
            c = getConnection();
            // Supprimer les entrées récursivement grâce aux "ON DELETE CASCADE"
            // Supprime dans l'ordre les lignes correspondantes des tables
            // OS_HISTORYSTEP_PREV, OS_CURRENTSTEP_PREV, OS_HISTORYSTEP,
            // OS_CURRENTSTEP
            String sql = "DELETE FROM " + entryTable + " WHERE " + entryId + " = ?";
            // Exécuter la requête
            stmt = c.prepareStatement(sql);
            // Positionner l'identifiant de l'instance du workflow
            stmt.setLong(1, idWorkflow);

            // Logger la requête
            _logger.debug(sql);

            // Vérifier si une ligne a bien été supprimée
            if (stmt.executeUpdate() != 1)
            {
                _logger.error("Unable to remove a workflow instance in database");
            }
        }
        catch (SQLException ex)
        {
            _logger.error("Error while removing a workflow instance", ex);
        }
        finally
        {
            cleanup(c, stmt, null);
        }
    }

    public void clearHistory(long idWorkflow)
    {
        Connection c = null;
        PreparedStatement stmt = null;

        try
        {
            // Récupérer la connexion du pool
            c = getConnection();
            // Supprimer les entrées récursivement grâce aux "ON DELETE CASCADE"
            // Supprime dans l'ordre les lignes correspondantes des tables
            // OS_HISTORYSTEP_PREV, OS_CURRENTSTEP_PREV, OS_HISTORYSTEP
            String sql = "DELETE FROM " + historyTable + " WHERE " + stepEntryId + " = ?";
            // Exécuter la requête
            stmt = c.prepareStatement(sql);
            // Positionner l'identifiant de l'instance du workflow
            stmt.setLong(1, idWorkflow);

            // Logger la requête
            _logger.debug(sql);

            // Supprimer les éventuelles lignes
            stmt.executeUpdate();
        }
        catch (SQLException e)
        {
            _logger.error("Error while clearing history steps from an workflow instance", e);
        }
        finally
        {
            cleanup(c, stmt, null);
        }
    }

    @Override
    public List query(WorkflowExpressionQuery e) throws StoreException
    {
        Expression expression = e.getExpression();

        if (expression instanceof PropertySetExpression)
        {
            return JDBCPropertySet.query(ds, expression);
        }
        else if (expression instanceof NestedExpression)
        {
            NestedExpression expr = (NestedExpression) expression;

            // Vérifier qu'il n'y ai pas autre chose que du PropertySet
            // imbriqué avant de l'envoyer au JDBCPropertySet
            if (JDBCPropertySet.isPropertySetExpressionsNested(expr))
            {
                return JDBCPropertySet.query(ds, expression);
            }

            return super.query(e);
        }
        else
        {
            return super.query(e);
        }
    }
    
    @Override
    protected long getNextEntrySequence(Connection c) throws SQLException
    {
        // DATABASE_MYSQL - DATABASE_DERBY
        String databaseType = getDatabaseType();
        if (ConnectionHelper.DATABASE_MYSQL.equals(databaseType) || ConnectionHelper.DATABASE_DERBY.equals(databaseType))
        {
            PreparedStatement stmt = null;
            ResultSet rset = null;
            
            try
            {
                stmt = c.prepareStatement(_entrySequenceIncrement);
                stmt.executeUpdate();
                rset = c.createStatement().executeQuery(_entrySequenceRetrieve);
                
                rset.next();
                
                long id = rset.getLong(1);
                
                return id;
            }
            finally
            {
                cleanup(null, stmt, rset);
            }
        }
        else
        {
            return super.getNextEntrySequence(c);
        }
    }

    @Override
    protected long getNextStepSequence(Connection c) throws SQLException
    {
        // DATABASE_MYSQL - DATABASE_DERBY
        String databaseType = getDatabaseType();
        if (ConnectionHelper.DATABASE_MYSQL.equals(databaseType) || ConnectionHelper.DATABASE_DERBY.equals(databaseType))
        {
            PreparedStatement stmt = null;
            ResultSet rset = null;
            
            try
            {
                stmt = c.prepareStatement(_stepSequenceIncrement);
                stmt.executeUpdate();
                rset = c.createStatement().executeQuery(_stepSequenceRetrieve);
                
                rset.next();
                
                long id = rset.getLong(1);
                
                return id;
            }
            finally
            {
                cleanup(null, stmt, rset);
            }
        }
        else
        {
            return super.getNextEntrySequence(c);
        }
    }
}
