/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.support;

import java.util.Collections;
import java.util.Properties;

import javax.jcr.Repository;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.environment.Context;
import org.apache.excalibur.source.SourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workflow.AbstractAmetysWorkflow;
import org.ametys.plugins.workflow.AmetysWorkflowFactory;
import org.ametys.plugins.workflow.SimpleConfiguration;
import org.ametys.plugins.workflow.repository.WorkflowAwareAmetysObject;
import org.ametys.plugins.workflow.store.AmetysObjectWorkflowStore;
import org.ametys.plugins.workflow.store.GenericWorkflowStore;

import com.opensymphony.workflow.FactoryException;
import com.opensymphony.workflow.StoreException;
import com.opensymphony.workflow.TypeResolver;
import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.WorkflowContext;
import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.loader.WorkflowFactory;
import com.opensymphony.workflow.spi.WorkflowStore;

/**
 * Component able to provide several workflow objects.
 * <ul>
 * <li>The generic workflow: A standalone workflow which can be used for various
 * purposes. All entries are stored under a global workflow root node in the JCR
 * repository.
 * <li>Dedicated ametys object workflows: A workflow specific to an ametys
 * object. The workflow is stored under the ametys object node the JCR
 * repository.
 * </ul>
 */
public class WorkflowProvider implements Component, Serviceable, Contextualizable, Initializable, Disposable 
{
    /** Avalon role. */
    public static final String ROLE = WorkflowProvider.class.getName();
    
    /** Logger available to subclasses. */
    protected static Logger _logger = LoggerFactory.getLogger(WorkflowProvider.class);

    /** Service manager. */
    protected ServiceManager _manager;
    
    /** Current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The repository */
    protected Repository _repository;
    
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    /** Cocoon context */
    protected Context _cocoonContext;
    
    /** Workflow context */
    protected WorkflowContext _workflowContext;
    
    /** Workflow factory */
    protected WorkflowFactory _workflowFactory;
    
    /** Type resolver */
    protected TypeResolver _typeResolver;
    
    /** Generic workflow instance */
    protected GenericWorkflow _genericWorkflow;

    /** The source resolver */
    protected SourceResolver _resolver;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _ametysObjectResolver;
    
    public void contextualize(org.apache.avalon.framework.context.Context ctx) throws ContextException
    {
        _cocoonContext = (Context) ctx.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _currentUserProvider = (CurrentUserProvider) _manager.lookup(CurrentUserProvider.ROLE);
        _repository = (Repository) _manager.lookup("javax.jcr.Repository");
        _workflowHelper = (WorkflowHelper) _manager.lookup(WorkflowHelper.ROLE);
        _resolver = (SourceResolver) _manager.lookup(SourceResolver.ROLE);
        _workflowFactory = (WorkflowFactory) _manager.lookup(AmetysWorkflowFactory.ROLE);
        _ametysObjectResolver = (AmetysObjectResolver) _manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void initialize() throws Exception
    {
        // Standard workflow context
        _workflowContext = new WorkflowContext()
        {
            public String getCaller()
            {
                return UserIdentity.userIdentityToString(_currentUserProvider.getUser());
            }
            
            public void setRollbackOnly()
            {
                // Transaction not supported
            }
        };

        // Ametys workflow factory
        try
        {
            _workflowFactory.init(new Properties());
            _workflowFactory.initDone();
        }
        catch (FactoryException e)
        {
            throw new RuntimeException("Unable to init workflow factory", e);
        }
        
        // Standard type resolver
        _typeResolver = new AvalonTypeResolver(_manager);
    }
    
    public void dispose()
    {
        _manager = null;
        _currentUserProvider = null;
        _cocoonContext = null;
        
        _workflowContext = null;
        _workflowFactory = null;
        _typeResolver = null;
        _genericWorkflow = null;
    }
    
    /**
     * Workflow factory getter
     * @return The workflow factory
     */
    WorkflowFactory getWorkflowFactory()
    {
        return _workflowFactory;
    }
    
    /**
     * Provides the generic workflow
     * @return The generic workflow
     */
    public Workflow getGenericWorkflow()
    {
        if (_genericWorkflow == null)
        {
            // Setup the generic workflow
            GenericWorkflowStore store = _createGenericWorkflowStore();
            try
            {
                store.init(Collections.EMPTY_MAP);
            }
            catch (StoreException e)
            {
                throw new RuntimeException("Unable to init workflow store", e);
            }
            
            _genericWorkflow = new GenericWorkflow(_workflowHelper, _workflowContext);
            
            _genericWorkflow.setConfiguration(new SimpleConfiguration(_workflowFactory, store));
            _genericWorkflow.setResolver(_typeResolver);
        }
        
        return _genericWorkflow;
    }
    
    /**
     * Provide the generic workflow store
     * @return The generic workflow store
     */
    protected GenericWorkflowStore _createGenericWorkflowStore()
    {
        return new GenericWorkflowStore(_repository);
    }
    
    /**
     * The Generic worklfow 
     */
    public static class GenericWorkflow extends AbstractAmetysWorkflow
    {
        GenericWorkflow(WorkflowHelper workflowHelper, WorkflowContext workflowContext)
        {
            super(workflowHelper, workflowContext);
        }
    }
    
    /**
     * Provide a local workflow to an Ametys object which do not preserve history on workflow complete
     * Must be used to initialize a workflow that will create an ametys object.
     * {@link AmetysObjectWorkflow#getAmetysObject()} could then be used to directly retrieves the ametys object.
     * @return The local workflow
     */
    public AmetysObjectWorkflow getAmetysObjectWorkflow()
    {
        return getAmetysObjectWorkflow(null, false);
    }
    
    /**
     * Provide a local workflow to an Ametys object
     * Must be used to initialize a workflow that will create an ametys object.
     * {@link AmetysObjectWorkflow#getAmetysObject()} could then be used to directly retrieves the ametys object.
     * @param preserveHistory true if the history should be preserve when workflow is complete
     * @return The local workflow
     */
    public AmetysObjectWorkflow getAmetysObjectWorkflow(boolean preserveHistory)
    {
        return getAmetysObjectWorkflow(null, preserveHistory);
    }
    
    /**
     * Provide a local workflow to an Ametys object which do not preserve history on workflow complete
     * @param ametysObject The ametys object (can be null in case of initialization)
     * @return the local workflow to an Ametys object
     */
    public AmetysObjectWorkflow getAmetysObjectWorkflow(WorkflowAwareAmetysObject ametysObject)
    {
        return getAmetysObjectWorkflow(ametysObject, false);
    }
    
    /**
     * Provide a local workflow to an Ametys object
     * @param ametysObject The ametys object (can be null in case of initialization)
     * @param preserveHistory true if the history steps should be preserve when workflow is complete
     * @return the local workflow to an Ametys object
     */
    public AmetysObjectWorkflow getAmetysObjectWorkflow(WorkflowAwareAmetysObject ametysObject, boolean preserveHistory)
    {
        AmetysObjectWorkflowStore store = _createAmetysObjectWorkflowStore(ametysObject, preserveHistory);
        
        try
        {
            store.init(Collections.EMPTY_MAP);
        }
        catch (StoreException e)
        {
            throw new RuntimeException("Unable to init workflow store", e);
        }
        
        AmetysObjectWorkflow ametysObjectWorkflow = new AmetysObjectWorkflow(_workflowHelper, _workflowContext, store);
        ametysObjectWorkflow.setConfiguration(new SimpleConfiguration(_workflowFactory, store));
        ametysObjectWorkflow.setResolver(_typeResolver);
        
        return ametysObjectWorkflow;
    }
    
    /**
     * Provide an ametys object workflow store instance
     * @param ametysObject The ametys object bound to this store (can be null in case of initialization)
     * @param preserveHistory true if the history steps should be preserve when workflow is complete
     * @return the local workflow store of an Ametys object
     */
    protected AmetysObjectWorkflowStore _createAmetysObjectWorkflowStore(WorkflowAwareAmetysObject ametysObject, boolean preserveHistory)
    {
        return new AmetysObjectWorkflowStore(_repository, ametysObject, _ametysObjectResolver, preserveHistory);
    }
    
    /**
     * Local workflow to an ametys object
     */
    public static class AmetysObjectWorkflow extends AbstractAmetysWorkflow
    {
        /** The ametys object */
        protected AmetysObjectWorkflowStore _workflowStore;
        
        AmetysObjectWorkflow(WorkflowHelper workflowHelper, WorkflowContext workflowContext, AmetysObjectWorkflowStore store)
        {
            super(workflowHelper, workflowContext);
            _workflowStore = store;
        }
        
        /**
         * Ametys object getter
         * @param <A> A WorkflowAwareAmetysObject class
         * @return The ametys object
         */
        @SuppressWarnings("unchecked")
        public <A extends WorkflowAwareAmetysObject> A getAmetysObject()
        {
            return (A) _workflowStore.getAmetysObject();
        }
        
        /**
         * Delete a workflow instance
         * @param wId The id of workflow instance
         * @throws WorkflowException if failed to delete workflow instance
         */
        public void removeWorkflow (long wId) throws WorkflowException
        {
            _workflowStore.removeEntry(wId);
        }

        @Override
        public String getWorkflowName(long id)
        {
            String workflowName = super.getWorkflowName(id);
            if (workflowName == null)
            {
                log.error("Can't get workflow name of " + getAmetysObject().toString());
            }
            return workflowName;
        }
    }
    
    /**
     * Provides an external workflow
     * @param workflowStoreRole The component role of the workflow store to use.
     * @return The external workflow
     */
    public Workflow getExternalWorkflow(String workflowStoreRole)
    {
        WorkflowStore workflowStore;
        try
        {
            workflowStore = (WorkflowStore) _manager.lookup(workflowStoreRole);
        }
        catch (ServiceException e)
        {
            throw new RuntimeException(String.format("Unable to lookup workflow store for role '%s'.", workflowStoreRole), e);
        }
        
        ExternalWorkflow workflow = new ExternalWorkflow(_workflowHelper, _workflowContext, workflowStore);
        
        workflow.setConfiguration(new SimpleConfiguration(_workflowFactory, workflowStore));
        workflow.setResolver(_typeResolver);
            
        return workflow;
    }
    
    /**
     * Provides an external workflow
     * @param workflowStore An (already initialized) workflow store
     * @return The external workflow
     */
    public Workflow getExternalWorkflow(WorkflowStore workflowStore)
    {
        ExternalWorkflow workflow = new ExternalWorkflow(_workflowHelper, _workflowContext, workflowStore);
        
        workflow.setConfiguration(new SimpleConfiguration(_workflowFactory, workflowStore));
        workflow.setResolver(_typeResolver);
            
        return workflow;
    }
    
    /**
     * The External worklfow 
     */
    public static class ExternalWorkflow extends AbstractAmetysWorkflow
    {
        /** The external workflow store */
        protected WorkflowStore _store;
        
        ExternalWorkflow(WorkflowHelper workflowHelper, WorkflowContext workflowContext, WorkflowStore workflowStore)
        {
            super(workflowHelper, workflowContext);
            _store = workflowStore;
        }
    }
}
