/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.support;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Session;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;
import org.w3c.dom.Document;

import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.workflow.component.WorkflowLanguageManager;
import org.ametys.plugins.workflow.dao.WorkflowsDAO;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.DescriptorFactory;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Helper for workflow in session
 */
public class WorkflowSessionHelper extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** The Avalon role */
    public static final String ROLE = WorkflowSessionHelper.class.getName();
    
    private static final String __WORKFLOW_NAMES = "workflowNames";
    
    /** The workflow language manager */
    protected WorkflowLanguageManager _workflowLanguageManager;
    
    /** The context */
    protected Context _context;
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    /** The helper for i18n translations and catalogs */
    protected I18nHelper _i18nHelper;
    
    /** The source resolver */
    protected SourceResolver _sourceResolver;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _workflowHelper = (WorkflowHelper) smanager.lookup(WorkflowHelper.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _workflowLanguageManager = (WorkflowLanguageManager) smanager.lookup(WorkflowLanguageManager.ROLE);
        _i18nHelper = (I18nHelper) smanager.lookup(I18nHelper.ROLE);
        _sourceResolver = (SourceResolver) smanager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * True if the workflow has non saved changes
     * @param workflowName unique name of workflow
     * @return <code>true</code> if there is a session for this workflow
     */
    public boolean hasChanges(String workflowName)
    {
        return _getWorkflowToSave(workflowName, false) != null;
    }
    
    /**
     * Get the label for current workflow
     * @param workflowName the workflow's unique name
     * @return the label as String or i18nizableText
     */
    public I18nizableText getWorkflowLabel(String workflowName)
    {
        WorkflowToSave workflowToSave = _getWorkflowToSave(workflowName, false);
        if (workflowToSave != null)
        {
            I18nizableText workflowLabelKey = _i18nHelper.getWorkflowLabelKey(workflowName);
            Map<String, String> workflowLabels = workflowToSave.getTranslations().get(workflowLabelKey);
            if (workflowLabels != null)
            {
                // Get the translated workflow label because it may be a temporary translation before saving
                return new I18nizableText(workflowLabels.get(_workflowLanguageManager.getCurrentLanguage()));
            }
        }
        return _workflowHelper.getWorkflowLabel(workflowName);
    }
    
    /**
     * Get the translations for current workflow names
     * @param workflowName the workflow's unique name
     * @return a map with available transitions, key is language, value is translation
     */
    public Map<String, String> getWorkflowLabelTranslations(String workflowName)
    {
        I18nizableText workflowLabelKey = _i18nHelper.getWorkflowLabelKey(workflowName);
        return Optional.of(workflowName)
            .map(_getSession()::getAttribute)
            .map(WorkflowToSave.class::cast)
            .map(WorkflowToSave::getTranslations)
            .map(translations -> translations.get(workflowLabelKey))
            .orElseGet(HashMap::new);
    }
    
    /**
     * Get translations map for current workflow
     * @param workflowName the current workflow's id
     * @return the map of traslations, key is i18n key, value is map of language, translation
     */
    public Map<I18nizableText, Map<String, String>> getTranslations(String workflowName)
    {
        return Optional.ofNullable(_getWorkflowToSave(workflowName, false))
                .map(WorkflowToSave::getTranslations)
                .orElseGet(() -> _initTranslations(workflowName));
    }
    
    /**
     * Get the translation for this key, null if it doesn't exist
     * @param workflowName the worflow's unique id
     * @param i18nKey the key to get translation for
     * @return the multilangual translation
     */
    public Map<String, String> getTranslation(String workflowName, I18nizableText i18nKey)
    {
        return getTranslations(workflowName).get(i18nKey);
    }
    
    /**
     * Update the translations map
     * @param workflowName the workflow's unique name
     * @param labelKey i18n key
     * @param translations the map of translations (language, translation) to put in the map
     */
    public void updateTranslations(String workflowName, I18nizableText labelKey, Map<String, String> translations)
    {
        WorkflowToSave workflowToSave = _getWorkflowToSave(workflowName, true);
        workflowToSave.updateTranslations(labelKey, translations);
        _updateWorkflowToSave(workflowToSave);
    }
    
    /**
     * Remove an entry from the translations map
     * @param workflowName the workflow's unique name
     * @param key the key of the entry to remove
     */
    public void removeTranslation(String workflowName, I18nizableText key)
    {
        WorkflowToSave workflowToSave = _getWorkflowToSave(workflowName, true);
        workflowToSave.getTranslations().remove(key);
        _updateWorkflowToSave(workflowToSave);
    }
    
    /**
     * Get the list of workflow names in use
     * @return a set of the names
     */
    @SuppressWarnings("unchecked")
    public Set<String> getWorkflowNames()
    {
        Session session = _getSession();
        if (session.getAttribute(__WORKFLOW_NAMES) != null)
        {
            return (Set<String>) session.getAttribute(__WORKFLOW_NAMES);
        }
        else
        {
            String[] names = _workflowHelper.getWorkflowNames();
            return new HashSet<>(Arrays.asList(names));
        }
    }

    /**
     * Update the session workflow names list with current workflow name
     * @param workflowDescriptor the current workflow descriptor
     */
    public void updateWorkflowNames(WorkflowDescriptor workflowDescriptor)
    {
        Set<String> workflowNames = getWorkflowNames();
        workflowNames.add(workflowDescriptor.getName());
        _getSession().setAttribute(__WORKFLOW_NAMES, workflowNames);
    }
    
    /**
     * Get workflow to save from session if exist
     * @param workflowName the current workflow's id
     * @param initIfNull <code>true</code> to create a new WorkflowToSave if session doesn't have it
     * @return the workflow to save
     */
    private WorkflowToSave _getWorkflowToSave(String workflowName, boolean initIfNull)
    {
        Session session = _getSession();
        
        if (session.getAttribute(workflowName) == null && initIfNull)
        {
            _initSession(workflowName);
        }
        return (WorkflowToSave) session.getAttribute(workflowName);
    }

    private void _initSession(String workflowName)
    {
        WorkflowDescriptor workflow = _cloneWorkflowDescriptor(workflowName);
        _initSession(workflowName, workflow);
    }
    
    private void _initSession(String workflowName, WorkflowDescriptor workflow)
    {
        Session session = _getSession();
        
        Map<I18nizableText, Map<String, String>> translations = _initTranslations(workflowName);
        session.setAttribute(workflowName, new WorkflowToSave(workflow, translations));
    }

    /**
     * Init map of translations with messages in current catalogs 
     * @param workflowName name of the workflow
     * @return the map of existing translations
     */
    private Map<I18nizableText, Map<String, String>> _initTranslations(String workflowName)
    {
        Map<I18nizableText, Map<String, String>> translations = new HashMap<>();
        Set<String> languages = _workflowLanguageManager.getLanguages();
        String defaultLanguage = _i18nHelper.getI18nDefaultLanguage();
        String workflowCatalog = _workflowHelper.getWorkflowCatalog(workflowName);
        String prefixCatalogPath = _i18nHelper.getPrefixCatalogLocation(workflowCatalog);
        
        // Read application existing catalogs
        for (String language : languages)
        {
            String catalogPath = prefixCatalogPath + (workflowCatalog.equals("application") && language.equals(defaultLanguage) ? "" : ("_" + language)) + ".xml";
            Map<I18nizableText, String> catalog = _i18nHelper.readI18nCatalog(new HashMap<>(), catalogPath, workflowCatalog);
            
            // Add catalog to translations map
            for (Entry<I18nizableText, String> entry : catalog.entrySet())
            {
                String value = entry.getValue();
                if (StringUtils.isNotEmpty(value))
                {
                    I18nizableText i18nKey = entry.getKey();
                    Map<String, String> translation  = translations.computeIfAbsent(i18nKey, __ -> new HashMap<>());
                    translation.put(language, value);
                }
            }
        }
        
        return translations;
    }

    /**
     * Update session's workflow to save 
     * @param workflowToSave the workflow to save to put in the session
     */
    private void _updateWorkflowToSave(WorkflowToSave workflowToSave)
    {
        _getSession().setAttribute(workflowToSave.getWorkflow().getName(), workflowToSave);
    }
    
    
    /**
     * Get workflow descriptor from session if exist, from xml if not
     * @param workflowName the current workflow's id
     * @param initSessionIfNull <code>true</code> to create a new WorkflowToSave if session doesn't have it
     * @return the workflow
     */
    public WorkflowDescriptor getWorkflowDescriptor(String workflowName, boolean initSessionIfNull)
    {
        WorkflowToSave workflowToSave = _getWorkflowToSave(workflowName, initSessionIfNull);
        return workflowToSave != null
                ? workflowToSave.getWorkflow()
                : _cloneWorkflowDescriptor(workflowName);
    }
    
    /**
     * Return a copy of a workflow descriptor
     * @param workflowName identifier of the workflow
     * @return the copy of the workflow
     */
    private WorkflowDescriptor _cloneWorkflowDescriptor(String workflowName)
    {
        try
        {
            WorkflowDescriptor workflowDescriptor = _workflowHelper.getWorkflowDescriptor(workflowName);
            if (workflowDescriptor == null)
            {
                return null;
            }
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder docB = dbf.newDocumentBuilder();
            
            try (InputStream is = new ByteArrayInputStream(workflowDescriptor.asXML().getBytes(StandardCharsets.UTF_8)))
            {
                Document doc = docB.parse(is);
                WorkflowDescriptor workflowDescriptorCopy = DescriptorFactory.getFactory().createWorkflowDescriptor(doc.getDocumentElement());
                workflowDescriptorCopy.setName(workflowName);
                
                return workflowDescriptorCopy;
            }
        }
        catch (Exception e) 
        {
            getLogger().error("An error occurred creating clone of workflow descriptor with name {}", workflowName, e);
        }
        
        return null;
    }
    
    /**
     * Init session cache with workflow descriptor changes
     * @param workflow the workflow to init
     */
    public void initWorkflowDescriptor(WorkflowDescriptor workflow)
    {
        _initSession(workflow.getName(), workflow);
    }
    
    /**
     * Update session cache with workflow descriptor changes
     * @param workflow the workflow to update
     */
    public void updateWorkflowDescriptor(WorkflowDescriptor workflow)
    {
        String workflowName = workflow.getName();
        WorkflowToSave workflowToSave = _getWorkflowToSave(workflowName, true);
        workflowToSave.setWorkflow(workflow);
        _updateWorkflowToSave(workflowToSave);
    }
    
    /**
     * Duplicate a workflow
     * @param newWorkflowName the unique name for the duplicate
     * @param labels the duplicate's labels
     * @param duplicatedWorkflowName the duplicated workflow's name
     * @return the duplicate 
     */
    public WorkflowDescriptor duplicateWorkflow(String newWorkflowName, Map<String, String> labels, String duplicatedWorkflowName)
    {
        WorkflowDescriptor workflowCloned = _cloneWorkflowDescriptor(duplicatedWorkflowName);
        workflowCloned.setName(newWorkflowName);
        workflowCloned.getMetaAttributes().put("user", true);

        //add meta to new workflow to prevent reinit
        workflowCloned.getMetaAttributes().put(WorkflowsDAO.META_NEW_WORKFLOW, true);
        
        //save workflow name in session
        _initSession(newWorkflowName, workflowCloned);
        updateWorkflowNames(workflowCloned);
        
        WorkflowToSave workflowToSave = _getWorkflowToSave(newWorkflowName, true);
        workflowToSave.updateTranslations(_i18nHelper.getWorkflowLabelKey(newWorkflowName), labels);
        workflowToSave.setWorkflow(workflowCloned);
        workflowToSave.setCloned(true);
        Map<I18nizableText, I18nizableText> clonedKeys = new HashMap<>();
        _cloneStepTranslations(workflowCloned, workflowCloned.getSteps(), clonedKeys);
        Set<Integer> transitionIds = _workflowHelper.getAllActions(workflowCloned);
        _cloneTransitionsTranslations(workflowCloned, transitionIds, clonedKeys);
        workflowToSave.setClonedKeys(clonedKeys);
        return workflowCloned;
    }
    
    /**
     * Remove workflow from session
     * @param workflowName unique name of the workflow
     */
    public void deleteSession(String workflowName)
    {
        _getSession().removeAttribute(workflowName);
    }
    
    private Session _getSession()
    {
        Request request = ContextHelper.getRequest(_context);
        return request.getSession(true);
    }
    
    /**
     * Clone the workflow's actions names's i18n translations into the application.xml translations
     * @param workflow the cloned workflow
     * @param actionIds the list of the workflow's actions ids
     */
    private void _cloneTransitionsTranslations(WorkflowDescriptor workflow, Set<Integer> actionIds, Map<I18nizableText, I18nizableText> clonedKeys)
    {
        Map<I18nizableText, Map<String, String>> applicationTranslations = getTranslations(workflow.getName());
        for (Integer actionId : actionIds)
        {
            ActionDescriptor action = workflow.getAction(actionId);
            I18nizableText actionLabelKey = new I18nizableText("application", action.getName());
            I18nizableText newActionLabelKey = _i18nHelper.generateI18nKey(workflow.getName(), "action", action.getId());
            clonedKeys.put(newActionLabelKey, actionLabelKey);
            _updateApplicationTranslations(applicationTranslations, actionLabelKey, newActionLabelKey);
            
            action.setName(newActionLabelKey.toString());
        }
    }
    
    /**
     * Clone the workflow's steps names's i18n translations into the application.xml translations
     * @param workflow the cloned workflow
     * @param steps the workflow's steps
     */
    private void _cloneStepTranslations(WorkflowDescriptor workflow, List<StepDescriptor> steps, Map<I18nizableText, I18nizableText> clonedKeys)
    {
        Map<I18nizableText, Map<String, String>> applicationTranslations = getTranslations(workflow.getName());
        for (StepDescriptor step : steps)
        {
            I18nizableText stepLabelKey = new I18nizableText("application", step.getName());
            I18nizableText newStepLabelKey = _i18nHelper.generateI18nKey(workflow.getName(), "step", step.getId());
            clonedKeys.put(newStepLabelKey, stepLabelKey);
            _updateApplicationTranslations(applicationTranslations, stepLabelKey, newStepLabelKey);
            
            step.setName(newStepLabelKey.toString());
        }
    }

    /**
     * Update translations from application.xml with the new label key
     * @param applicationTranslations the map of translations from application.xml: key is elementName, value is a map of language, translation
     * @param initialLabelKey the initial key before copy
     * @param newLabelKey the new key for the duplicated element
     */
    private void _updateApplicationTranslations(Map<I18nizableText, Map<String, String>> applicationTranslations, I18nizableText initialLabelKey, I18nizableText newLabelKey)
    {
        //Translations are already in application catalog
        Set<String> languages = _workflowLanguageManager.getLanguages();
        if (applicationTranslations.containsKey(initialLabelKey))
        {
            applicationTranslations.put(newLabelKey, applicationTranslations.get(initialLabelKey));
        }
        else
        {
            //.. search for translations
            Map<String, String> actionTranslations = new HashMap<>();
            for (String language : languages)
            {
                String translation = _i18nUtils.translate(initialLabelKey, language);
                if (StringUtils.isNotBlank(translation))
                {
                    actionTranslations.put(language, translation);
                }
            }
            if (!actionTranslations.isEmpty())
            {
                applicationTranslations.put(newLabelKey, actionTranslations);
            }
        }
    }
    
    /**
     * Clone kernel workflow images into WEB-INF/workflow_resources with new labels as names
     * @param workflowName the current workflow name
     */
    public void cloneImages(String workflowName)
    {
        WorkflowToSave workflowToSave = _getWorkflowToSave(workflowName, false);
        if (workflowToSave.isCloned())
        {
            Map<I18nizableText, I18nizableText> clonedKeys = workflowToSave.getClonedKeys();
            for (Entry<I18nizableText, I18nizableText> entry : clonedKeys.entrySet())
            {
                I18nizableText elementLabel = entry.getValue();
                I18nizableText newElementLabel = entry.getKey();
                for (String size : WorkflowHelper.ICON_SIZES)
                {
                    _copyImage(elementLabel, newElementLabel, size);
                }
            }
            workflowToSave.setCloned(false);
        }
        
    }
    
    /**
     * Get the original label key used for this element if the workflow is a clone 
     * @param workflowName the current workflow name
     * @param label the current label key of the workflow element
     * @return the original label if exist, the same label if not
     */
    public I18nizableText getOldLabelKeyIfCloned(String workflowName, I18nizableText label)
    {
        WorkflowToSave workflowToSave = _getWorkflowToSave(workflowName, false);
        if (workflowToSave != null && workflowToSave.isCloned())
        {
            Map<I18nizableText, I18nizableText> clonedKeys = workflowToSave.getClonedKeys();
            return clonedKeys.containsKey(label) ? clonedKeys.get(label) : label;
        }
        else
        {
            return label;
        }
    }
    
    private void _copyImage(I18nizableText elementLabel, I18nizableText newElementLabel, String size)
    {
        Source imageSource = null;
        FileSource clonedImageSource = null;
        try
        {
            imageSource = _sourceResolver.resolveURI(_workflowHelper.getElementIconURI(elementLabel, size));
            if (imageSource.exists())
            {
                clonedImageSource = (FileSource) _sourceResolver.resolveURI(_workflowHelper.getElementIconURI(newElementLabel, size));
                File clonedImage = clonedImageSource.getFile();
                FileUtils.createParentDirectories(clonedImage);
                try (
                        InputStream is = imageSource.getInputStream();
                        OutputStream os = new FileOutputStream(clonedImage);
                    )
                {
                    IOUtils.copy(is, os);
                }
            }
        }
        catch (IOException  e)
        {
            getLogger().error("An error occured while copying existing i18n catalog", e);
        }
        finally
        {
            _sourceResolver.release(imageSource);
            _sourceResolver.release(clonedImageSource);
        }
    }
}
