/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.icsreader;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.slf4j.Logger;

import org.ametys.cms.tag.Tag;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.calendar.events.EventsFilterHelper;
import org.ametys.plugins.calendar.icsreader.ical4j.AmetysParameterFactorySupplier;
import org.ametys.plugins.calendar.icsreader.ical4j.DefaultComponentFactorySupplier;
import org.ametys.plugins.calendar.icsreader.ical4j.DefaultPropertyFactorySupplier;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.LogEnabled;

import net.fortuna.ical4j.data.CalendarBuilder;
import net.fortuna.ical4j.data.CalendarParserFactory;
import net.fortuna.ical4j.data.ContentHandlerContext;
import net.fortuna.ical4j.filter.predicate.PeriodRule;
import net.fortuna.ical4j.model.Calendar;
import net.fortuna.ical4j.model.Component;
import net.fortuna.ical4j.model.DateList;
import net.fortuna.ical4j.model.DateTime;
import net.fortuna.ical4j.model.Period;
import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.TimeZoneRegistryFactory;
import net.fortuna.ical4j.model.component.CalendarComponent;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.parameter.Value;
import net.fortuna.ical4j.model.property.DateProperty;
import net.fortuna.ical4j.model.property.DtEnd;
import net.fortuna.ical4j.model.property.DtStart;
import net.fortuna.ical4j.model.property.RRule;

/**
 * Read a distant ICS file for a certain number of events in the future
 */
public class IcsReader implements Serviceable, org.apache.avalon.framework.component.Component, Initializable, LogEnabled
{
    /** The Avalon role. */
    public static final String ROLE = IcsReader.class.getName();

    private static final String __ICS_CACHE_ID = IcsReader.class.getName() + "$icsCache";

    private static final int __ICS_CONNECTION_TIMEOUT = 2000;
    private static final int __ICS_READ_TIMEOUT = 10000;
    
    /** logger */
    protected Logger _logger;

    private AbstractCacheManager _abstractCacheManager;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _abstractCacheManager = (AbstractCacheManager) smanager.lookup(AbstractCacheManager.ROLE);
    }

    /**
     * Get a list of events from an ics file
     * @param url url of the ics file
     * @param dateRange range of dates to fetch
     * @param nbEvents number of events to read
     * @param maxFileSize max ics file size (in bytes)
     * @return a List of {@link VEvent}
     */
    public IcsEvents getEventList(String url, EventsFilterHelper.DateTimeRange dateRange, Long nbEvents, Long maxFileSize)
    {
        getLogger().debug("Fetch ics url : {}", url);
        CacheKey cacheKey = new CacheKey(url, dateRange, nbEvents, maxFileSize);

        Cache<CacheKey, IcsEvents> cache = getIcsCache();
        return cache.get(cacheKey, key -> _getEventList(url, dateRange, nbEvents, maxFileSize));
    }

    /**
     * Get a list of events from an ics file, without trying the cache
     * @param url url of the ics file
     * @param dateRange range of dates to fetch
     * @param nbEvents number of events to read
     * @param maxFileSize max ics file size (in bytes)
     * @return a List of {@link VEvent}
     */
    protected IcsEvents _getEventList(String url, EventsFilterHelper.DateTimeRange dateRange, Long nbEvents, Long maxFileSize)
    {
        try
        {
            long fileSize = getFileSize(url);
            if (fileSize > maxFileSize)
            {
                getLogger().warn("ICS File is too big : {}", url);
                return new IcsEvents(url, null, IcsEvents.Status.OVERSIZED);
            }

            URL icsUrl = new URL(url);

            HttpURLConnection connection = (HttpURLConnection) icsUrl.openConnection();
            connection.setConnectTimeout(__ICS_CONNECTION_TIMEOUT);
            connection.setReadTimeout(__ICS_READ_TIMEOUT);

            String userInfo = icsUrl.getUserInfo();
            if (userInfo != null)
            {
                String basicAuth = "Basic " + Base64.getEncoder().encodeToString(userInfo.getBytes(StandardCharsets.UTF_8));
                connection.setRequestProperty("Authorization", basicAuth);
            }

            try (InputStream body = connection.getInputStream())
            {
                ContentHandlerContext contentHandlerContext = new ContentHandlerContext().withParameterFactorySupplier(new AmetysParameterFactorySupplier())
                                                                                         .withPropertyFactorySupplier(new DefaultPropertyFactorySupplier())
                                                                                         .withComponentFactorySupplier(new DefaultComponentFactorySupplier());

                CalendarBuilder builder = new CalendarBuilder(CalendarParserFactory.getInstance().get(),
                                                              contentHandlerContext,
                                                              TimeZoneRegistryFactory.getInstance().createRegistry());

                Calendar calendar = builder.build(body);

                getLogger().debug("Calendar is built for url : {}", url);

                List<CalendarComponent> components = calendar.getComponents(Component.VEVENT);

                Collection<CalendarComponent> componentList;

                if (dateRange != null)
                {
                    Period period = new Period(new DateTime(DateUtils.asDate(dateRange.fromDate())), new DateTime(DateUtils.asDate(dateRange.untilDate())));
                    componentList = components.stream().filter(new PeriodRule<>(period)).collect(Collectors.toList());
                }
                else
                {
                    componentList = components;
                }

                getLogger().debug("Calendar is filtered for url : {}", url);

                List<VEvent> eventList = new ArrayList<>();
                Long nbEventsRemaining = nbEvents;
                for (CalendarComponent calendarComponent : componentList)
                {
                    if (nbEventsRemaining > 0)
                    {
                        if (calendarComponent instanceof VEvent)
                        {
                            eventList.add((VEvent) calendarComponent);
                            nbEventsRemaining--;
                        }
                    }
                    else
                    {
                        break;
                    }
                }
                getLogger().debug("List is generated for url : {}", url);

                return new IcsEvents(url, eventList);
            }
        }
        catch (Exception e)
        {
            getLogger().error("Error while reading ics with url = '" + url + "'", e);
            return new IcsEvents(url, null, IcsEvents.Status.ERROR);
        }
    }

    public void initialize() throws Exception
    {
        System.setProperty("ical4j.unfolding.relaxed", "true");
        System.setProperty("net.fortuna.ical4j.timezone.cache.impl", "net.fortuna.ical4j.util.MapTimeZoneCache");

        Long cacheTtlConf = Config.getInstance().getValue("org.ametys.plugins.calendar.ics.reader.cache.ttl");
        Long cacheTtl = (long) (cacheTtlConf != null && cacheTtlConf.intValue() > 0 ? cacheTtlConf.intValue() : 60);

        Duration duration = Duration.ofMinutes(cacheTtl);

        _abstractCacheManager.createMemoryCache(__ICS_CACHE_ID,
                new I18nizableText("plugin.calendar", "CALENDAR_SERVICE_AGENDA_ICS_CACHE_LABEL"),
                new I18nizableText("plugin.calendar", "CALENDAR_SERVICE_AGENDA_ICS_CACHE_DESC"),
                false, // ical4j events crash the api that calculates the size, sorry
                duration);
    }

    private Cache<CacheKey, IcsEvents> getIcsCache()
    {
        return _abstractCacheManager.get(__ICS_CACHE_ID);
    }

    /**
     * Try to get the size of the file, and download it if needed
     * @param url the url to get
     * @return size in octet, or -1 if error
     * @throws IOException Something went wrong
     */
    private long getFileSize(String url) throws IOException
    {
        getLogger().debug("Start to try to determine size of the file : {}", url);

        URL icsUrl = new URL(url);

        HttpURLConnection connection = (HttpURLConnection) icsUrl.openConnection();
        connection.setConnectTimeout(__ICS_CONNECTION_TIMEOUT);
        connection.setReadTimeout(__ICS_READ_TIMEOUT);

        long nbByte = connection.getContentLengthLong();

        if (nbByte < 0)
        {
            try (InputStream flux = connection.getInputStream())
            {
                getLogger().debug("Unable to get size from header, we download the file : {}", url);
                nbByte = 0;
                while (flux.read() != -1)
                {
                    nbByte++;
                }
            }
        }
        getLogger().debug("End of estimation of the size of the file, {} bytes : {}", nbByte, url);
        return nbByte;
    }

    /**
     * Get a list of {@link LocalDate} covered by this event
     * @param event the event to test
     * @param dateRange the dates to check, can be null but this will return null
     * @param tag the tag used for this ICS
     * @return a list of {@link LocalDate} for the days in which this event appears, or null if nothing matches
     */
    public List<LocalVEvent> getEventDates(VEvent event, EventsFilterHelper.DateTimeRange dateRange, Tag tag)
    {
        List<LocalVEvent> result = new ArrayList<>();

        Property rRuleProperty = event.getProperty(Property.RRULE);
        if (rRuleProperty instanceof RRule)
        {
            Property startProperty = event.getProperty(Property.DTSTART);
            Property endProperty = event.getProperty(Property.DTEND);
            if (startProperty instanceof DtStart && endProperty instanceof DtEnd)
            {
                if (dateRange != null)
                {
                    DtStart dtStart = (DtStart) startProperty;
                    DtEnd dtEnd = (DtEnd) endProperty;
                    long eventDurationInMs = dtEnd.getDate().getTime() - dtStart.getDate().getTime();

                    RRule rRule = (RRule) rRuleProperty;
                    net.fortuna.ical4j.model.Date periodeStart = new net.fortuna.ical4j.model.Date(DateUtils.asDate(dateRange.fromDate()));
                    net.fortuna.ical4j.model.Date periodeEnd = new net.fortuna.ical4j.model.Date(DateUtils.asDate(dateRange.untilDate()));

                    DateList dates = rRule.getRecur().getDates(dtStart.getDate(), periodeStart, periodeEnd, Value.DATE_TIME);

                    for (net.fortuna.ical4j.model.Date startDate : dates)
                    {
                        long eventEnd = startDate.getTime() + eventDurationInMs;
                        Date endDate;
                        if (dtEnd.getDate() instanceof DateTime)
                        {
                            endDate = new DateTime(eventEnd);
                        }
                        else
                        {
                            endDate = new Date(Instant.ofEpochMilli(eventEnd).minus(1, ChronoUnit.DAYS).toEpochMilli());
                        }
                        result.add(new LocalVEvent(event, startDate, endDate, tag));
                    }
                }
                else
                {
                    getLogger().debug("Impossible to get the lest of events without a date range, it can be too much");
                }
            }
        }
        else
        {
            Date startDate = _getEventDateTime(event, Property.DTSTART);
            Date endDate = _getEventDateTime(event, Property.DTEND);

            // If no dates, it can not be displayed.
            // If one date is missing, consider the other equals
            if (startDate == null && endDate == null)
            {
                return result;
            }
            else if (startDate == null)
            {
                startDate = endDate;
            }
            else if (endDate == null)
            {
                endDate = startDate;
            }

            result.add(new LocalVEvent(event, startDate, endDate, tag));
        }
        return result;
    }

    /**
     * Return a string representing the start/end date of an event, or null if no start/end date was found
     * @param event the event to read
     * @param property a {@link Property} to check ( {@link Property#DTSTART} or  {@link Property#DTEND} )
     * @return a string representing the date
     */
    private Date _getEventDateTime(CalendarComponent event, String property)
    {
        Date result = null;
        Property checkedProperty = event.getProperty(property);
        if (checkedProperty instanceof DateProperty)
        {
            DateProperty checked = (DateProperty) checkedProperty;
            result = checked.getDate();

            // if we check the DTEND and it is just a DATE (not a DATETIME), we remove one day to it (because it is exclusive)
            if (result != null && Property.DTEND.equals(property) && !(result instanceof DateTime))
            {
                result = new Date(result.toInstant().minus(1, ChronoUnit.DAYS).toEpochMilli());
            }
        }
        return result;
    }

    public void setLogger(Logger logger)
    {
        _logger = logger;
    }

    private Logger getLogger()
    {
        return _logger;
    }

    /**
     * Object wrapper for ics events
     */
    public static class IcsEvents
    {
        /**
         * The status of the ics
         */
        public static enum Status
        {
            /** If the ics exceeds the authorized max size */
            OVERSIZED,
            /** If there is some erros reading ics url */
            ERROR,
            /** If the ics is OK */
            OK
        }

        private String _url;
        private List<VEvent> _events;
        private Status _status;
        private Tag _tag;

        /**
         * The constructor
         * @param url the url
         * @param events the list of event of the ics
         */
        public IcsEvents(String url, List<VEvent> events)
        {
            this(url, events, Status.OK);
        }

        /**
         * The constructor
         * @param url the url
         * @param events the list of event of the ics
         * @param status the status of the ics
         */
        public IcsEvents(String url, List<VEvent> events, Status status)
        {
            _url = url;
            _events = events;
            _tag = null;
            _status = status;
        }

        /**
         * Get the url of the ics
         * @return the url
         */
        public String getUrl()
        {
            return _url;
        }

        /**
         * <code>true</code> if the ics has events
         * @return <code>true</code> if the ics has events
         */
        public boolean hasEvents()
        {
            return _events != null && _events.size() > 0;
        }

        /**
         * Get the list of events of the ics
         * @return the list of events
         */
        public List<VEvent> getEvents()
        {
            return _events;
        }

        /**
         * Get the tag of the ics
         * @return the tag
         */
        public Tag getTag()
        {
            return _tag;
        }

        /**
         * Set the tag of the ics
         * @param tag the tag
         */
        public void setTag(Tag tag)
        {
            _tag = tag;
        }

        /**
         * Get the status of the ics
         * @return the status
         */
        public Status getStatus()
        {
            return _status;
        }
    }
}
