/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.icsreader.ical4j;

import java.net.URI;
import java.net.URISyntaxException;

import net.fortuna.ical4j.model.Content;
import net.fortuna.ical4j.model.Parameter;
import net.fortuna.ical4j.model.ParameterFactory;
import net.fortuna.ical4j.util.Strings;
import net.fortuna.ical4j.util.Uris;

/**
 * From specification:
 *
 * <pre>
 *  Parameter Name:  EMAIL
 *
 *  Purpose:  To specify an email address that is used to identify or
 *  contact an organizer or attendee.
 *
 *  Format Definition:  This property parameter is defined by the
 *  following notation:
 *
 *  emailparam = "EMAIL" "=" param-value
 *
 *  Description:  This property parameter MAY be specified on "ORGANIZER"
 *  or "ATTENDEE" properties.  This property can be used in situations
 *  where the calendar user address value of "ORGANIZER" and
 *  "ATTENDEE" properties is not likely to be an identifier that
 *  recipients of scheduling messages could use to match the calendar
 *  user with, for example, an address book entry.  The value of this
 *  property is an email address that can easily be matched by
 *  recipients.  Recipients can also use this value as an alternative
 *  means of contacting the calendar user via email.  If a recipient's
 *  calendar user agent allows the recipient to save contact
 *  information based on the "ORGANIZER" or "ATTENDEE" properties,
 *  those calendar user agents SHOULD use any "EMAIL" property
 *  parameter value for the email address of the contact over any
 *  mailto: calendar user address specified as the value of the
 *  property.  Calendar user agents SHOULD NOT include an "EMAIL"
 *  property parameter when its value matches the calendar user
 *  address specified as the value of the property.
 *
 *  Example:
 *
 *  ATTENDEE;CN=Cyrus Daboo;EMAIL=cyrus@example.com:mailto:opaque-toke
 *  n-1234@example.com
 * </pre>
 */
// FIXME this class has been taken from the ical4j's github repo, in a branch without the javax.mail dependency
// to be removed as soon as ical4j 4.0 is released
// ical4j is licensed under the BSD license
@SuppressWarnings("all")
public class Email extends Parameter {

    private static final long serialVersionUID = 1L;

    private static final String PARAMETER_NAME = "EMAIL";

    private final URI address;

    public Email(String address) throws URISyntaxException {
        this(Uris.create(Strings.unquote(address)));
    }

    public Email(URI address) {
        super(PARAMETER_NAME, new Factory());
        this.address = address;
    }

    public URI getAddress() {
        return address;
    }

    @Override
    public String getValue() {
        return Uris.decode(Strings.valueOf(getAddress()));
    }

    public static class Factory extends Content.Factory implements ParameterFactory<Email> {
        private static final long serialVersionUID = 1L;

        public Factory() {
            super(PARAMETER_NAME);
        }

        @Override
        public Email createParameter(final String value) {
            try {
                return new Email(value);
            } catch (URISyntaxException e) {
                throw new IllegalArgumentException(e);
            }
        }
    }
}
