/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.search;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.search.advanced.AbstractTreeNode;
import org.ametys.cms.search.advanced.TreeLeaf;
import org.ametys.cms.tag.CMSTag;
import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.calendar.icsreader.IcsEventHelper;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstance;
import org.ametys.web.frontoffice.search.instance.model.FOSearchCriterion;
import org.ametys.web.frontoffice.search.requesttime.SearchComponent;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.frontoffice.search.requesttime.impl.SaxFormSearchComponent;
import org.ametys.web.frontoffice.search.requesttime.impl.SearchComponentHelper;
import org.ametys.web.frontoffice.search.requesttime.input.SearchUserInputs;
import org.ametys.web.repository.page.ZoneItem;

/**
 * {@link SearchComponent} for saxing form for calendar search service
 * Reactivate default {@link SaxFormSearchComponent} (disabled by "disableDefaultSax" parameter)
 */
public class CalendarSaxFormSearchComponent extends SaxFormSearchComponent
{
    /** Helper to get ICS events */
    protected IcsEventHelper _icsEventHelper;
    /** The tags provider */
    protected TagProviderExtensionPoint _tagProviderEP;
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    /** The user preferences manager */
    protected UserPreferencesManager _userPrefsManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _icsEventHelper = (IcsEventHelper) smanager.lookup(IcsEventHelper.ROLE);
        _tagProviderEP = (TagProviderExtensionPoint) smanager.lookup(TagProviderExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _userPrefsManager = (UserPreferencesManager) smanager.lookup(UserPreferencesManager.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public boolean supports(SearchComponentArguments args)
    {
        return CalendarSearchService.isActive(args);
    }
    
    @Override
    public void execute(SearchComponentArguments args) throws Exception
    {
        ContentHandler contentHandler = args.contentHandler();
        SearchServiceInstance serviceInstance = args.serviceInstance();
        List<FOSearchCriterion> nonStaticCriteria = serviceInstance.getCriterionTree()
                .map(AbstractTreeNode::getFlatLeaves)
                .orElseGet(Collections::emptyList)
                .stream()
                .map(TreeLeaf::getValue)
                .filter(c -> !c.getMode().isStatic())
                .collect(Collectors.toList());
        Parameters parameters = args.generatorParameters();
        
        XMLUtils.startElement(contentHandler, "form");
        
        XMLUtils.startElement(contentHandler, "fields");
        Map<String, Object> contextualParameters = SearchComponentHelper.getSearchComponentContextualParameters(args);
        saxFormFields(contentHandler, parameters, nonStaticCriteria, contextualParameters, serviceInstance.computeCriteriaCounts());
        XMLUtils.endElement(contentHandler, "fields");
        
        SearchUserInputs userInputs = args.userInputs();
        
        Map<String, Object> userCriteria = userInputs.criteria();
        XMLUtils.startElement(contentHandler, "values");
        saxFormValues(contentHandler, parameters, nonStaticCriteria, userCriteria);
        XMLUtils.endElement(contentHandler, "values");
        
        String zoneItemId = args.serviceInstance().getId();
        ZoneItem zoneItem = (ZoneItem) _resolver.resolveById(zoneItemId);
        
        if (CalendarSearchService.hasTagCategories(args.serviceInstance()))
        {
            List<String> categoryNames = CalendarSearchService.getTagCategories(serviceInstance);
            Set<Tag> tagCategories = getTagCategories(categoryNames, args);
            tagCategories.addAll(_icsEventHelper.getIcsTags(zoneItem, args.currentSite().getName()));
            saxTagFacet(contentHandler, tagCategories, args);
        }
        XMLUtils.endElement(contentHandler, "form");
    }
    
    /**
     * Get the configured tag categories
     * @param categoryNames The parent tag category
     * @param args the search arguments
     * @return the tags
     */
    protected Set<Tag> getTagCategories(List<String> categoryNames, SearchComponentArguments args)
    {
        Set<Tag> tags = new HashSet<>();
        
        if (categoryNames != null && !categoryNames.isEmpty())
        {
            Map<String, Object> params = Map.of("siteName", args.currentSite().getName());
            
            Set<CMSTag> tagCategories = categoryNames.stream()
                    .map(c -> _tagProviderEP.getTag(c, params))
                    .filter(Objects::nonNull)
                    .collect(Collectors.toSet());
            
            for (CMSTag cmsTag : tagCategories)
            {
                tags.addAll(_getChildTags(cmsTag));
            }
        }
        
        return tags;
    }
    
    private Set<Tag> _getChildTags(Tag tag)
    {
        Set<Tag> allTags = new LinkedHashSet<>();

        Map<String, ? extends Tag> childTagsMap = tag.getTags();

        if (childTagsMap != null)
        {
            Collection<? extends Tag> childTags = childTagsMap.values();
            allTags.addAll(childTags);
            
            for (Tag child : childTags)
            {
                allTags.addAll(_getChildTags(child));
            }
        }

        return allTags;
    }
    
    /**
     * Generate the list of tags
     * @param handler the content handler
     * @param tags the list of tags
     * @param args the search arguments
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxTagFacet(ContentHandler handler, Collection<Tag> tags, SearchComponentArguments args) throws SAXException
    {
        Set<String> selectedTags = new HashSet<>(CalendarSearchService.getSelectedTags(args));
        
        if (CalendarSearchService.saveUserPrefsEnabled(args) && CalendarSearchService.saveFilterInUserPrefs(args.serviceInstance()))
        {
            List<String> userTags = _getCurrentSearchUserTagsFacet(args.serviceInstance().getId());
            selectedTags.addAll(userTags);
        }
        
        XMLUtils.startElement(handler, "tag-facet");
        for (Tag tag : tags)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("selected",  String.valueOf(selectedTags.contains(tag.getName())));
            CalendarSearchService.saxTag(handler, attrs, tag);
        }
        XMLUtils.endElement(handler, "tag-facet");
    }
    
    /**
     * Get the current search user tags for the given service.
     * @param serviceId The service ID
     * @return the current user tags if the user is authenticated and have saved filters for this service. Otherwise, it returns <code>null</code>.
     */
    protected List<String> _getCurrentSearchUserTagsFacet(String serviceId)
    {
        try
        {
            UserIdentity user = _currentUserProvider.getUser();
            if (user != null)
            {
                Map<String, String> userPreferences = _userPrefsManager.getUnTypedUserPrefs(user, "calendar-search-" + serviceId, new HashMap<>());

                if (!userPreferences.isEmpty())
                {
                    return _jsonUtils.convertJsonToList(userPreferences.get("search.calendar.tags")).stream()
                        .filter(String.class::isInstance)
                        .map(String.class::cast)
                        .filter(StringUtils::isNotBlank)
                        .collect(Collectors.toList());
                }
            }
            else
            {
                getLogger().debug("There is no authenticated user.");
            }
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occured while getting the user preferences.", e);
        }
        
        return List.of();
    }
    
}
