/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.events;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Date;

import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Helper class that provides a method to check if a date is between two others.
 */
public final class EventHelper implements LogEnabled, Serviceable
{    
    /** The Ametys object resolver */
    protected static AmetysObjectResolver _resolver;
    /** The extension point for content types */
    protected static ContentTypeExtensionPoint _contentTypeEP;
    
    private static Logger _logger;

    @Override
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    /**
     * Tests if the given date is either:
     * - equal to the first, if there is no end date.
     * - between the two dates, if there is a start and end date.
     * @param dateStr the date to test.
     * @param startStr start of the interval.
     * @param endStr end of the interval.
     * @return true if the date is either equal to the start date or between the start and end date.
     */
    public static boolean isBetween(String dateStr, String startStr, String endStr)
    {
        boolean between = false;
        LocalDate date = LocalDate.parse(dateStr, DateTimeFormatter.ISO_LOCAL_DATE);
        LocalDate start = LocalDate.parse(startStr, DateTimeFormatter.ISO_LOCAL_DATE);
        
        if (StringUtils.isBlank(endStr))
        {
            between = date.equals(start);
        }
        else
        {
            LocalDate end = LocalDate.parse(endStr, DateTimeFormatter.ISO_LOCAL_DATE);
            between = (date.isAfter(start) || date.equals(start)) && (date.isBefore(end) || date.equals(end));
        }
        
        return between;
    }
    
    /**
     * Tests if the given event occurs in the given month.
     * @param monthStartStr The month first day, cannot be null or blank.
     * @param eventStartStr The event start date, can be null or blank only if the end date is set.
     * @param eventEndStr The event end date, can be null or blank only if the end date is set.
     * @return true if the event occurs in the given month.
     */
    public static boolean inMonth(String monthStartStr, String eventStartStr, String eventEndStr)
    {
        boolean overlaps = false;
        
        Date myDate = DateUtils.parse(monthStartStr);
        ZonedDateTime monthStart = myDate.toInstant().atZone(ZoneId.systemDefault());
        ZonedDateTime monthEnd = monthStart.toLocalDate().atStartOfDay(monthStart.getZone()).plusMonths(1);
        
        if (StringUtils.isNotBlank(eventStartStr) && StringUtils.isNotBlank(eventEndStr))
        {
            ZonedDateTime eventStart = DateUtils.parse(eventStartStr).toInstant().atZone(ZoneId.systemDefault());
            ZonedDateTime eventEnd = DateUtils.parse(eventEndStr).toInstant().atZone(ZoneId.systemDefault());
            
            // check that start is before end to avoid an unwanted exception
            try
            {
                // If eventStart equals to eventEnd and equals to monthStart, month#overlaps(Interval) will return false. In our case, we want to consider it actually does.
                overlaps = eventStartStr.equals(eventEndStr) ? _contains(monthStart, monthEnd, eventStart)
                                                             : _overlaps(monthStart, monthEnd, eventStart, eventEnd);
            }
            catch (IllegalArgumentException e)
            {
                // The end is before the start
                overlaps = false;
                _logger.error(String.format("Invalid dates of event: the end date (%s) must be greater or equal to the start date (%s). The event will be ignored in calendar view.", eventEndStr, eventStartStr), e);
            }
        }
        else if (StringUtils.isNotBlank(eventStartStr))
        {
            ZonedDateTime eventStart = (DateUtils.parse(eventStartStr)).toInstant().atZone(ZoneId.systemDefault());
            overlaps = _contains(monthStart, monthEnd, eventStart);
        }
        else if (StringUtils.isNotBlank(eventEndStr))
        {
            ZonedDateTime eventEnd = (DateUtils.parse(eventEndStr)).toInstant().atZone(ZoneId.systemDefault());
            overlaps = _contains(monthStart, monthEnd, eventEnd);
        }
        
        return overlaps;
    }
    
    /**
     * Test if an event is between 2 dates
     * @param start start date
     * @param end end date
     * @param event event to check
     * @return true if the event is between both dates
     */
    private static boolean _contains(ZonedDateTime start, ZonedDateTime end, ZonedDateTime event)
    {
        ZonedDateTime realStart;
        ZonedDateTime realEnd;
        if (start.isBefore(end))
        {
            realStart = start;
            realEnd = end;
        }
        else
        {
            realStart = end;
            realEnd = start;
        }
        return (realStart.isBefore(event) || realStart.isEqual(event)) && realEnd.isAfter(event);
    }
    
    /**
     * Test if 2 period of time overlap
     * @param start1 start of 1st period
     * @param end1 end of 1st period
     * @param start2 start of 2nd period
     * @param end2 end of 2nd period
     * @return true if both period overlaps
     */
    private static boolean _overlaps(ZonedDateTime start1, ZonedDateTime end1, ZonedDateTime start2, ZonedDateTime end2)
    {
        ZonedDateTime realStart1;
        ZonedDateTime realEnd1;
        if (start1.isBefore(end1))
        {
            realStart1 = start1;
            realEnd1 = end1;
        }
        else
        {
            realStart1 = end1;
            realEnd1 = start1;
        }
        ZonedDateTime realStart2;
        ZonedDateTime realEnd2;
        if (start2.isBefore(end2))
        {
            realStart2 = start2;
            realEnd2 = end2;
        }
        else
        {
            realStart2 = end2;
            realEnd2 = start2;
        }
        return (realStart1.isBefore(realEnd2) || realStart1.isEqual(realEnd2)) && realEnd1.isAfter(realStart2);
    }
    
    /**
     * Get the next day
     * @param dateStr the zoned date time
     * @return the next day as iso zoned date time
     */
    public static String nextDay(String dateStr)
    {
        ZonedDateTime zonedDateTime = DateUtils.parseZonedDateTime(dateStr);
        zonedDateTime = zonedDateTime.plusDays(1);
        
        return DateUtils.zonedDateTimeToString(zonedDateTime);
    }
    
    /**
     * Get the next day
     * @param dateStr the zoned date time
     * @param format the output format: "date" for ISO local date or "basicDate" to basic ISO date.
     * @return the next day at requested output format
     */
    public static String nextDay(String dateStr, String format)
    {
        String nextDayStr = "";
        
        ZonedDateTime date = ZonedDateTime.parse(dateStr, DateUtils.getISODateTimeFormatter());
        
        ZonedDateTime nextDay = date.plusDays(1);
        
        if ("basicDate".equals(format))
        {
            nextDayStr = DateTimeFormatter.BASIC_ISO_DATE.format(nextDay);
        }
        else
        {
            nextDayStr = DateTimeFormatter.ISO_LOCAL_DATE.format(nextDay);
        }
        
        return nextDayStr;
    }
    
    /**
     * Tests if the given attribute for the given content is of type datetime
     * @param contentId The id of the content
     * @param attributeName The attribute name
     * @return true if the given attribute for the given content is of type datetime
     */
    public static boolean isDatetime(String contentId, String attributeName)
    {
        Content content = _resolver.resolveById(contentId);
        String[] contentTypeIds = content.getTypes();
        
        for (String contentTypeId : contentTypeIds)
        {
            ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
            
            if (contentType.hasModelItem(attributeName))
            {
                ModelItem definition = contentType.getModelItem(attributeName);
                if (definition instanceof ElementDefinition)
                {
                    ElementType type = ((ElementDefinition) definition).getType();
                    return ModelItemTypeConstants.DATETIME_TYPE_ID.equals(type.getId());
                }
                else
                {
                    // The given attribute is a group item
                    return false;
                }
            }
        }
        
        return false;
    }
    
}
