/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.calendar.helpers;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.environment.Request;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.JSONUtils;
import org.ametys.web.WebHelper;

/**
 * helper to manage calendar categories
 */
public class CalendarCategoriesHelper implements Component, Serviceable
{
    /** The component role. */
    public static final String ROLE = CalendarCategoriesHelper.class.getName();
    /**
     * Parameter name for the zone item id
     */
    protected static final String ZONEITEMID = "zoneItemId";
    
    /**
     * user pref id where the categories should be stored
     */
    protected static final String USER_PREF_ID = "plugin-calendar-categories";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The user preferences */
    protected UserPreferencesManager _userPref;
    
    /** Json utils */
    protected JSONUtils _jsonUtils;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userPref = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }

    /**
     * Set the user preferences for this calendar
     * @param request the request from the service
     * @return true if the preference was set
     * @throws UserPreferencesException impossible to set the preference
     */
    public boolean setUserCategories(Request request) throws UserPreferencesException
    {
        UserIdentity identity = _currentUserProvider.getUser();
        
        if (identity != null)
        {
            String siteName = WebHelper.getSiteName(request);
            String zoneItemId = request.getParameter(ZONEITEMID);
            String categorieString = request.getParameter("categories");
            List<String> categories = new ArrayList<>();
            if (categorieString != null)
            {
                categories.add(categorieString);
            }
            else
            {
                String[] categoriesString = request.getParameterValues("categories[]");
                categories.addAll(Arrays.asList(categoriesString));
            }
            
            return setUserCategories(identity, siteName, zoneItemId, categories);
        }
        else
        {
            return false;
        }
    }
    
    /**
     * Set the user preferences for this calendar
     * @param siteName Site name
     * @param zoneItemId Zone Item Id where the service is put
     * @param categories categories to set
     * @return true if the preference was set
     * @throws UserPreferencesException impossible to set the preference
     */
    public boolean setUserCategories(String siteName, String zoneItemId, List<String> categories) throws UserPreferencesException
    {
        UserIdentity identity = _currentUserProvider.getUser();
        return setUserCategories(identity, siteName, zoneItemId, categories);
    }
    
    /**
     * Set the user preferences for this calendar
     * @param identity User
     * @param siteName Site name
     * @param zoneItemId Zone Item Id where the service is put
     * @param categories categories to set
     * @return true if the preference was set
     * @throws UserPreferencesException impossible to set the preference
     */
    public boolean setUserCategories(UserIdentity identity, String siteName, String zoneItemId, List<String> categories) throws UserPreferencesException
    {
        if (identity == null)
        {
            return false;
        }
        
        String jsonCategories = _jsonUtils.convertObjectToJson(categories);
        
        String jsonString = _userPref.getUserPreferenceAsString(identity, "/sites/" + siteName, Collections.emptyMap(), USER_PREF_ID);
        
        Map<String, Object> jsonMap = _jsonUtils.convertJsonToMap(jsonString);
        
        if (jsonMap == null)
        {
            jsonMap = new HashMap<>();
        }
        
        jsonMap.put(zoneItemId, jsonCategories);
        
        jsonString = _jsonUtils.convertObjectToJson(jsonMap);
        
        Map<String, String> preferences = new HashMap<>();
        preferences.put(USER_PREF_ID, jsonString);
        
        _userPref.setUserPreferences(identity, "/sites/" + siteName, Collections.emptyMap(), preferences);
        
        return true;
    }
    
    /**
     * Get the user preferences for this calendar
     * @param request the request from the service
     * @return a List of categories
     * @throws UserPreferencesException impossible to get the preference
     */
    public List<String> getUserCategories(Request request) throws UserPreferencesException
    {
        UserIdentity identity = _currentUserProvider.getUser();
        
        String siteName = WebHelper.getSiteName(request);
        String zoneItemId = request.getParameter(ZONEITEMID);
        
        return getUserCategories(identity, siteName, zoneItemId);
    }
    
    /**
     * Get the user preferences for this calendar
     * @param siteName Site name
     * @param zoneItemId Zone Item Id where the service is put
     * @return a List of categories
     * @throws UserPreferencesException impossible to get the preference
     */
    public List<String> getUserCategories(String siteName, String zoneItemId) throws UserPreferencesException
    {
        UserIdentity identity = _currentUserProvider.getUser();
        return getUserCategories(identity, siteName, zoneItemId);
    }
    
    /**
     * Get the user preferences for this calendar
     * @param identity User
     * @param siteName Site name
     * @param zoneItemId Zone Item Id where the service is put
     * @return a List of categories
     * @throws UserPreferencesException impossible to set the preference
     */
    public List<String> getUserCategories(UserIdentity identity, String siteName, String zoneItemId) throws UserPreferencesException
    {
        List<String> result = new ArrayList<>();
        if (identity != null)
        {
            String jsonString = _userPref.getUserPreferenceAsString(identity, "/sites/" + siteName, Collections.emptyMap(), USER_PREF_ID);
            
            Map<String, ? extends Object> jsonMap = _jsonUtils.convertJsonToMap(jsonString);
            
            if (jsonMap != null && jsonMap.containsKey(zoneItemId))
            {
                Object object = jsonMap.get(zoneItemId);
                if (object instanceof String)
                {
                    result = _jsonUtils.convertJsonToList((String) object).stream()
                            .filter(String.class::isInstance)
                            .map(String.class::cast)
                            .collect(Collectors.toList());
                }
                else if (object instanceof List<?>)
                {
                    result = ((List<?>) object).stream()
                    .filter(String.class::isInstance)
                    .map(String.class::cast)
                    .collect(Collectors.toList());
                }
            }
            else
            {
                result.add("all");
            }
        }
        else
        {
            result.add("all");
        }
        
        return result;
    }

}
