/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.enumerators;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeEnumerator;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;

/**
 * Enumerator on {@link ContentType} for the Agenda service.
 * Restrict content types to those which have all the attributes referenced by the <code>_metadataNames</code> list.
 */
public class CalendarContentTypesEnumerator extends ContentTypeEnumerator
{
    /** Map of mandatory attributes names and their type id. */
    protected Map<String, Set<String>> _mandatoryAttributes;
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        Configuration mandatoryAttributesConf = configuration.getChild("enumeration").getChild("custom-enumerator").getChild("mandatory-metadata", false);
        if (mandatoryAttributesConf == null)
        {
            _mandatoryAttributes = null;
        }
        else
        {
            _mandatoryAttributes = new HashMap<>();
            
            Configuration[] metadataRefConfs = mandatoryAttributesConf.getChildren("metadata-ref");
            for (Configuration metadataRefConf : metadataRefConfs)
            {
                String name = metadataRefConf.getValue(null);
                
                String[] types = StringUtils.split(metadataRefConf.getAttribute("type", "string"), ',');
                
                Set<String> attributeTypes = new HashSet<>(types.length);
                for (String type : types)
                {
                    attributeTypes.add(type.trim().toUpperCase());
                }
                
                _mandatoryAttributes.put(name, attributeTypes);
            }
            _mandatoryAttributes.remove(null);
        }
    }
    
    @Override
    public Map<String, I18nizableText> getTypedEntries() throws Exception
    {
        Map<String, I18nizableText> entries = new HashMap<>();
        Set<ContentType> contentTypes = getMatchingContentTypes();
        
        for (ContentType contentType : contentTypes)
        {
            // Build a Set containing the rejected mandatory attributes.
            // This means either the attribute does not exist for this content type, or the attribute type is not matching the expectations.
            Collection<String> rejectedAttributes = CollectionUtils.selectRejected(_mandatoryAttributes.keySet(), new MandatoryAttributePredicate(contentType));

            if (rejectedAttributes.isEmpty())
            {
                entries.put(contentType.getId(), contentType.getLabel());
            }
        }

        // All contents
        _handleAllOptionEntry(entries);
        
        return entries;
    }
    
    /**
     * Predicate that test if the requested attribute is in the content type and if its type is correct.
     */
    public class MandatoryAttributePredicate implements Predicate
    {
        private ContentType _contentType;
        
        /**
         * Build a {@code MandatoryAttributePredicate}
         * @param contentType The ContentType to check against.
         */
        protected MandatoryAttributePredicate(ContentType contentType)
        {
            _contentType = contentType;
        }
        
        @Override
        public boolean evaluate(Object object)
        {
            String attributeName = (String) object;
            return _isDefined(attributeName) && _isTypeCorrect(attributeName); 
        }
        
        private boolean _isDefined(String attributeName)
        {
            return _contentType.hasModelItem(attributeName);
        }
        
        private boolean _isTypeCorrect(String attributeName)
        {
            ModelItem modelItem = _contentType.getModelItem(attributeName);
            if (modelItem instanceof ElementDefinition)
            {
                String modelItemType = ((ElementDefinition) modelItem).getType().getId().toUpperCase();
                return _mandatoryAttributes.get(attributeName).contains(modelItemType);
            }
            else
            {
                return false;
            }
        }
    }
}
